/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.lang.internal.ant;

import java.io.File;

import org.apache.tools.ant.Project;
import org.apache.tools.ant.taskdefs.optional.ssh.Scp;

import protoj.core.internal.AntTarget;

/**
 * A convenience class for performing a secure copy. Use the constructors to
 * specify the minimal and most widely anticipated configuration and the
 * <code>initXXX</code> methods for the less common configuration options.
 * 
 * @author Ashley Williams
 * 
 */
public final class ScpTask {
	private AntTarget target = new AntTarget("protoj-scp");
	private Scp scp;

	/**
	 * Please consult the man page for scp for information about these
	 * parameters.
	 * 
	 * @param user
	 * @param host
	 * @param localFile
	 * @param dest
	 */
	public ScpTask(String user, String host, File localFile, String dest) {
		scp = new Scp();
		scp.setTaskName("scp");
		target.addTask(scp);
		scp.setTodir(user + "@" + host + ":" + dest);
		scp.setLocalFile(localFile.getAbsolutePath());
		scp.setTrust(true);
	}

	/**
	 * Please consult the man page for scp for information about these
	 * parameters.
	 * 
	 * @param toDir
	 * @param localFile
	 */
	public ScpTask(String toDir, File localFile) {
		scp = new Scp();
		scp.setTaskName("scp");
		target.addTask(scp);
		scp.setTodir(toDir);
		scp.setLocalFile(localFile.getAbsolutePath());
		scp.setTrust(true);
	}

	/**
	 * Enables logging to the specified log file at Project.MSG_INFO level.
	 * 
	 * @param logFile
	 */
	public void initLogging(File logFile) {
		target.initLogging(logFile, Project.MSG_INFO);
	}

	/**
	 * Call this method to when using key authentication.
	 * 
	 * @param passphrase
	 * @param keyFile
	 */
	public void initKeyAuthentication(String passphrase, String keyFile) {
		scp.setPassphrase(passphrase);
		scp.setKeyfile(keyFile);
	}

	/**
	 * Call this method when using simple password authentication.
	 * 
	 * @param password
	 */
	public void initPasswordAuthentication(String password) {
		scp.setPassword(password);
	}

	/**
	 * The underlying ant task.
	 * 
	 * @return
	 */
	public Scp getScp() {
		return scp;
	}

	public void execute() {
		target.execute();
	}

}
