package protoj.core;

/**
 * Value object that represents a single instruction to be carried out. An
 * example of an instruction is: "mycommand -opt1 val1 -opt2 val2" where the
 * instruction name is "mycommand" and the instruction opts
 * is"-opt1 val1 -opt2 val2".
 * <p>
 * Typically many of these are contained in an {@link InstructionChain} instance
 * and represent the result of parsing a chain of instructions specified at the
 * command line.
 * 
 * @author Ashley Williams
 * 
 */
public final class Instruction {

	/**
	 * The name of the opts instruction
	 */
	private static final String OPTS = "opts";

	/**
	 * The name of the init instruction.
	 */
	private static final String INIT = "init";

	/**
	 * See {@link #getName()}.
	 */
	private String name;

	/**
	 * See {@link #getOpts()}.
	 */
	private String opts;

	@Override
	public String toString() {
		return super.toString();
	}

	/**
	 * See {@link Instruction}.
	 * 
	 * @param name
	 *            the name of the instruction
	 * @param opts
	 *            if there are no options then specify a null or an empty string
	 *            - both result in an empty string being returned from
	 *            {@link #getOpts()}.
	 */
	public Instruction(String name, String opts) {
		this.name = name.trim();
		this.opts = opts == null ? "" : opts.trim();
	}

	/**
	 * Returns a textual representation of this instruction suitable for the
	 * command line.
	 * 
	 * @return
	 */
	public String getText() {
		return new String(name + " " + opts).trim();
	}

	/**
	 * The name of the instruction.
	 * 
	 * @return
	 */
	public String getName() {
		return name;
	}

	/**
	 * The unparsed options for the instruction. Returns an empty string for no
	 * options.
	 * 
	 * @return
	 */
	public String getOpts() {
		return opts;
	}

	/**
	 * Returns true if this instruction has an options set.
	 * 
	 * @return
	 */
	public boolean hasOpts() {
		return opts.length() > 0;
	}

	/**
	 * Returns whether or not this instruction is the one that provides
	 * application initialization parameters , i.e. the INIT instruction.
	 * 
	 * @return
	 */
	public boolean isInitInstruction() {
		return INIT.equals(getName());
	}

	/**
	 * Returns whether or not this instruction is the one that provides system
	 * property options, i.e. the OPTS instruction. This is the name of the
	 * special instruction, so don't confuse it with the options to an
	 * instruction.
	 * 
	 * @return
	 */
	public boolean isOptsInstruction() {
		return OPTS.equals(getName());
	}

}