/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.core;

import java.io.File;
import java.io.FilenameFilter;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;

import org.apache.commons.configuration.CompositeConfiguration;
import org.apache.commons.configuration.Configuration;
import org.apache.commons.configuration.PropertiesConfiguration;
import org.apache.commons.configuration.SystemConfiguration;

import protoj.core.internal.CoreProject;

/**
 * Responsible for managing the available properties. The property values can be
 * obtained simply by querying the reference returned by {@link #getConfig()}.
 * <p>
 * Alternatively consider calling {@link PropertyInfo#getValue()}, which
 * provides for customized default behavior. These info objects can be added by
 * calling {@link #addInfo(String, String, String, String)} and are used for
 * example in ProtoJ help.
 * 
 * @author Ashley Williams
 * 
 */
public final class PropertyStore {
	/**
	 * See {@link #getCommands()}.
	 */
	private HashMap<String, PropertyInfo> properties = new HashMap<String, PropertyInfo>();

	/**
	 * The directory where the properties files are stored.
	 */
	private File confDir;

	/**
	 * See {@link #getConfig()}.
	 */
	private Configuration config;

	/**
	 * Creates an instance with the owning project.
	 * 
	 * @param parent
	 */
	public PropertyStore(CoreProject parent) {
		this.confDir = parent.getLayout().getConfDir();
		this.config = createConfig();
	}

	/**
	 * Creates a properties helper that is configured to look for property
	 * values first from any properties files immediately under the config
	 * directory and secondly from java.lang.System.
	 * 
	 * @return
	 */
	@SuppressWarnings("unchecked")
	private Configuration createConfig() {
		final CompositeConfiguration config = new CompositeConfiguration();
		confDir.listFiles(new FilenameFilter() {
			public boolean accept(File dir, String name) {
				if (name.endsWith(".properties")) {
					File file = new File(dir, name);
					config.addConfiguration(new PropertiesConfiguration(file));
				}
				return true;
			}
		});
		config.addConfiguration(new SystemConfiguration());

		return config.interpolatedConfiguration();
	}

	/**
	 * Convenience method that creates a {@link PropertyInfo} instance with the
	 * given arguments and adds it to this store.
	 * 
	 * @param key
	 * @param missingDefault
	 * @param emptyDefault
	 * @param description
	 * @return
	 */
	public PropertyInfo addInfo(String key, String missingDefault,
			String emptyDefault, String description) {
		PropertyInfo property = new PropertyInfo(this, key, missingDefault,
				emptyDefault, description);
		properties.put(key, property);
		return property;
	}

	/**
	 * Read-only access to the added {@link PropertyInfo} instances.
	 * 
	 * @return
	 */
	public Collection<PropertyInfo> getInfos() {
		return Collections.unmodifiableCollection(properties.values());
	}

	/**
	 * Looks up the {@link PropertyInfo} with the given key.
	 * 
	 * @param key
	 * @return
	 */
	public PropertyInfo getInfo(String key) {
		return properties.get(key);
	}

	/**
	 * The instance responsible for obtaining property values. See
	 * {@link #createConfig()}.
	 * 
	 * @return
	 */
	public Configuration getConfig() {
		return config;
	}

}
