/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.core;

import protoj.core.internal.CoreProject;

/**
 * These are all the properties that are added to the {@link PropertyStore} on
 * behalf of the StandardProject.
 * 
 * @author Ashley Williams
 * 
 */
public final class StandardProperties {

	/**
	 * See {@link #getDebugProperty()}.
	 */
	private PropertyInfo debugProperty;

	/**
	 * See {@link #getLevelProperty()}.
	 */
	private PropertyInfo levelProperty;

	/**
	 * See {@link #getqDebugProperty()}.
	 */
	private PropertyInfo quietProperty;

	/**
	 * See {@link #getDebugSuspendProperty()}.
	 */
	private PropertyInfo debugSuspendProperty;

	/**
	 * See {@link #getDebugPortProperty()}.
	 */
	private PropertyInfo debugPortProperty;

	/**
	 * See {@link #getDebugConditionProperty()}.
	 */
	private PropertyInfo debugConditionProperty;

	/**
	 * See {@link #getGcskipProperty()}.
	 */
	private PropertyInfo gcskipProperty;

	/**
	 * See {@link #getScpskipProperty()}.
	 */
	private PropertyInfo scpskipProperty;

	/**
	 * See {@link #getStatusWindowProperty()}.
	 */
	private PropertyInfo statusWindowProperty;

	/**
	 * Create an instance with the owning parent project.
	 * 
	 * @param parent
	 */
	public StandardProperties(CoreProject parent) {
		this.levelProperty = parent
				.getPropertyStore()
				.addInfo(
						"protoj.level",
						"INFO",
						"INFO",
						"use this to configure application logging by setting to one of the log4j logging level constants");
		this.quietProperty = parent.getPropertyStore().addInfo("protoj.quiet",
				Boolean.FALSE.toString(), Boolean.TRUE.toString(),
				"set to true to stop most output to the console");
		this.debugProperty = parent.getPropertyStore().addInfo("protoj.debug",
				Boolean.FALSE.toString(), Boolean.TRUE.toString(),
				"set to true in order to debug a vm");
		this.debugSuspendProperty = parent
				.getPropertyStore()
				.addInfo("protoj.debug.suspend", Boolean.FALSE.toString(),
						Boolean.TRUE.toString(),
						"set to true to suspend the vm whilst waiting for a debugger connection");
		this.debugPortProperty = parent.getPropertyStore().addInfo(
				"protoj.debug.port", "11111", "11111",
				"set to a valid port number to be used when debugging a vm");
		this.debugConditionProperty = parent.getPropertyStore().addInfo(
				"protoj.debug.condition", "", "",
				getDebugConditionDescription());
		this.gcskipProperty = parent.getPropertyStore().addInfo(
				"protoj.gcskip", "true", "true", getGcskipDescription());
		this.scpskipProperty = parent.getPropertyStore().addInfo(
				"protoj.scpskip", "true", "true", getScpskipDescription());
		this.statusWindowProperty = parent.getPropertyStore().addInfo(
				"protoj.statusWindow", "false", "true",
				getStatusWindowDescription());
	}

	/**
	 * Set to true or false to to turn debugging on or off.
	 * 
	 * @return
	 */
	public PropertyInfo getDebugProperty() {
		return debugProperty;
	}

	/**
	 * One of the logging levels constant values as declared in the log4j Level
	 * class.
	 * 
	 * @return
	 */
	public PropertyInfo getLevelProperty() {
		return levelProperty;
	}

	/**
	 * Set to true or false to turn the console appender on or off.
	 * 
	 * @return
	 */
	public PropertyInfo getQuietProperty() {
		return quietProperty;
	}

	/**
	 * Set to true or false to switch suspend on or off for the debugging
	 * session.
	 * 
	 * @return
	 */
	public PropertyInfo getDebugSuspendProperty() {
		return debugSuspendProperty;
	}

	/**
	 * Set to a valid port number that the debugger should listen for
	 * connections on.
	 * 
	 * @return
	 */
	public PropertyInfo getDebugPortProperty() {
		return debugPortProperty;
	}

	/**
	 * Use to narrow down the many potential debug targets to a vm that contains
	 * a matching argument to its <code>main()</code> method. For example,
	 * specifying a value of <code>archive</code> will cause only the vm that
	 * performs the archive task to be debugged.
	 * 
	 * @return
	 */
	public PropertyInfo getDebugConditionProperty() {
		return debugConditionProperty;
	}

	/**
	 * Whether or not the upload to google code should occur.
	 * 
	 * @return
	 */
	public PropertyInfo getGcskipProperty() {
		return gcskipProperty;
	}

	/**
	 * Whether or not the remote copy via scp should occur.
	 * 
	 * @return
	 */
	public PropertyInfo getScpskipProperty() {
		return scpskipProperty;
	}

	/**
	 * Whether or not to display the colored window at the end of the build.
	 * 
	 * @return
	 */
	public PropertyInfo getStatusWindowProperty() {
		return statusWindowProperty;
	}

	private String getStatusWindowDescription() {
		StringBuilder builder = new StringBuilder();
		builder
				.append("Specify whether or not a window gets displayed indicating project");
		builder
				.append("\nsuccess (green) or failure (red). The application does not terminate");
		builder.append("\nunless the window is closed.");
		builder.append("\nThe two possible values are:");
		builder
				.append("\n\n   1. true: the windows gets diplayed, provided there is a gui system available.");
		builder.append("\n\n   2. false: no window gets displayed at all.");
		return builder.toString();
	}

	private String getScpskipDescription() {
		StringBuilder builder = new StringBuilder();
		builder
				.append("Specify whether or not the project tar file gets remotely copied during scp.");
		builder.append("\nThe two possible values are:");
		builder
				.append("\n\n   1. true: the tar file doesn't get copied and an informative message is logged.");
		builder.append("\n\n   2. false: the tar file gets copied normally.");
		return builder.toString();
	}

	/**
	 * Factoring out large description into its own method.
	 * 
	 * @return
	 */
	private String getDebugConditionDescription() {
		StringBuilder builder = new StringBuilder();
		builder
				.append("Specify criteria in order to narrow down which vm is to be debugged.");
		builder.append("\nThe following property value types are supported:");
		builder
				.append("\n\n   1. <command-name>: debug the vm started with the named command.");
		builder
				.append("\n\n      Example: '-Dprotoj.debug.condition=archive' targets the vm where jar building takes place.");
		builder
				.append("\n      Example: '-Dprotoj.debug.condition=' targets all command vm's.");
		builder
				.append("\n\n   2. junit-task: target the vm that is executes the junit tests.");
		builder
				.append("\n      Note that this does not target the parent 'test' vm that forks the junit vm");
		builder
				.append("\n      in the first place - to debug that vm use 'test' as in the first format.");
		builder
				.append("\n\n      Example: '-Dprotoj.debug.condition=junit-task'.");
		builder
				.append("\n\n   2. project: target the vm that configures the commands.");
		builder
				.append("\n      This is the class specified in the call to StandardProject.initBootstrap().");
		builder
				.append("\n\n      Example: '-Dprotoj.debug.condition=project'.");
		return builder.toString();
	}

	/**
	 * Factoring out large description into its own method.
	 * 
	 * @return
	 */
	private String getGcskipDescription() {
		StringBuilder builder = new StringBuilder();
		builder
				.append("Specify whether or not the project tar file gets uploaded to google code.");
		builder.append("\nThe two possible values are:");
		builder
				.append("\n\n   1. true: the tar file doesn't get posted and an informative message is logged.");
		builder.append("\n\n   2. false: the tar file gets posted normally.");
		return builder.toString();
	}
}
