/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.lang;

import protoj.core.ProjectLayout;
import protoj.core.internal.InformationException;
import protoj.lang.internal.ant.JunitTask;

/**
 * {@link StandardProject} helper that provides support for running junit tests.
 * 
 * @author Ashley Williams
 * 
 */
public final class JunitFeature {

	/**
	 * See {@link #getProject()}.
	 */
	private StandardProject project;

	private String memory;

	/**
	 * The individual property getter methods describe the parameters of the
	 * same includes.
	 * 
	 * @param project
	 * @param memory
	 */
	public JunitFeature(StandardProject project, String memory) {
		this.project = project;
		this.memory = memory;
	}

	/**
	 * The parent of this feature.
	 * 
	 * @return
	 */
	public StandardProject getProject() {
		return project;
	}

	/**
	 * The amount of memory to give to the forked vm that executes the unit
	 * tests. Uses standard java tool notation, e.g. "32m".
	 * 
	 * @return
	 */
	public String getMemory() {
		return memory;
	}

	/**
	 * Executes all the classes that contain 'Test' as part of their name.
	 * Delegates to {@link #junit(String, String)}.
	 */
	public void junit() {
		junit("**/*Test*.java", null);
	}

	/**
	 * Executes the junit tests filtering with the specified includes and
	 * excludes parameters. All tests gain the following system property which
	 * can prove useful in locating project directories and files:
	 * 
	 * <ul>
	 * <li>protoj.rootDir - the root directory for this project</li>
	 * <li>protoj.scriptName - the script location for this project</li>
	 * </ul>
	 * <p>
	 * A junit test failure will result in an {@link InformationException} being
	 * thrown.
	 * 
	 * @param includes
	 *            The ant pattern that determines which java classes will be
	 *            executed as junit tests. Set to null in order to not apply an
	 *            includes pattern
	 * @param excludes
	 *            The ant pattern that determines which java classes will not be
	 *            executed as junit tests. Set to null in order to not apply an
	 *            excludes pattern
	 */
	public void junit(String includes, String excludes) {
		ProjectLayout layout = project.getLayout();
		JunitTask junitTask = new JunitTask(layout.getJunitReportsDir(), layout
				.getJavaDir(), layout.getClasspathConfig(), getMemory(),
				includes, excludes);
		junitTask.initLogging(layout.getLogFile());
		junitTask.initJvmargs("-Dprotoj.rootDir=" + layout.getRootPath());
		junitTask.initJvmargs("-Dprotoj.scriptName=" + layout.getScriptName());
		try {
			junitTask.execute();
		} catch (Throwable e) {
			StringBuilder builder = new StringBuilder();
			builder
					.append("there was a problem running the junit tests, please check the junit reports directory");
			builder.append("\nfor more information: ");
			builder.append(layout.getJunitReportsDir().getAbsolutePath());
			throw new InformationException(builder.toString());
		}
	}
}