/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.lang.internal;

import joptsimple.OptionSet;
import joptsimple.OptionSpec;
import protoj.core.Command;
import protoj.core.CommandStore;
import protoj.lang.StandardProject;

/**
 * The command responsible for creating the project tar.
 * 
 * @author Ashley Williams
 * 
 */
public final class TarCommand {

	private final class Body implements Runnable {

		public void run() {
			OptionSet options = getDelegate().getOptions();
			boolean noSrc = options.has(getNosrc());
			boolean isRwx = options.has(getRwx());
			project.getArchiveFeature().getProjectArchive().createArchive(
					noSrc, isRwx);
		}
	}

	/**
	 * Provides the basic command functionality.
	 */
	private Command delegate;
	private OptionSpec<?> nosrc;
	private OptionSpec<?> rwx;
	private final StandardProject project;

	public TarCommand(StandardProject project) {
		this.project = project;
		CommandStore store = project.getCommandStore();

		StringBuilder description = new StringBuilder();
		description
				.append("Creates a compressed tar of the application ready for further distribution.");
		description.append("\nThe following command arguments are supported:");
		description
				.append("\n\n   1. -nosrc: when specified, the source directory is excluded, otherwise it is included.");
		description
				.append("\n\n   2. -rwx: when specified all directories and files are added to the tar with the least");
		description
				.append("\n      restrictive permissions, otherwise this override isn't applied.");
		description.append("\n\nExample: jonny$ ./");
		description.append(project.getLayout().getScriptName());
		description.append(" \"tar -nosrc -rwx\"");

		delegate = store.addCommand("tar", description.toString(), "16m",
				new Body());
		delegate.initAliases("gtar", "package");
		nosrc = delegate.getParser().accepts("nosrc");
		rwx = delegate.getParser().accepts("rwx");
	}

	public Command getDelegate() {
		return delegate;
	}

	public OptionSpec<?> getNosrc() {
		return nosrc;
	}

	public OptionSpec<?> getRwx() {
		return rwx;
	}
}
