/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.lang.internal.ajc;

import java.io.File;

import org.junit.Assert;

import protoj.core.ArgRunnable;
import protoj.core.ProjectLayout;
import protoj.lang.ScriptSession;
import protoj.lang.StandardProject;
import protoj.lang.internal.ProtoProject;
import protoj.lang.internal.ant.CommandTask;

/**
 * Executes a selection of commands against the ajc project and performs various
 * asserts against their results.
 * 
 * @author Ashley Williams
 * 
 */
public final class AjcSession {
	private static final String CLEAN = "clean";
	private static final String COMPILE = "compile";
	private static final String RETRIEVE = "retrieve";
	private static final String HELLO_AJC = "hello-ajc";
	private static final String HELP = "help";

	private StandardProject project;

	/**
	 * After this constructor call the physical demo project will have been
	 * created.
	 */
	public AjcSession() {
		ProtoProject protoProject = new ProtoProject();
		this.project = protoProject.createSampleProjectDelegate("ajc", true);
	}

	public void execute() {
		ScriptSession session = project.createScriptSession();
		ArgRunnableImplementation listener = new ArgRunnableImplementation();
		session.addCommand("compile", listener, COMPILE);
		session.addCommand("retrieve", listener, RETRIEVE);
		session.addCommand("help", listener, HELP);
		session.addCommand("hello-ajc", listener, HELLO_AJC);
		session.addCommand("clean", listener, CLEAN);
		session.execute();
	}

	private final class ArgRunnableImplementation implements
			ArgRunnable<ScriptSession> {
		public void run(ScriptSession session) {
			ProjectLayout layout = project.getLayout();
			CommandTask exec = session.getCurrentExec();
			String stdout = exec.getStdout();
			project.getLogger().debug(stdout);
			String log = project.getLayout().loadLog();

			String tag = session.getCurrentTag();
			if (tag.equals(RETRIEVE)) {
				File libDir = layout.getLibDir();
				File javadoc = new File(libDir, "junit-4.4-javadoc.jar");
				File sources = new File(libDir, "junit-4.4-sources.jar");
				File classes = new File(libDir, "junit-4.4.jar");
				Assert.assertTrue(javadoc.exists());
				Assert.assertTrue(sources.exists());
				Assert.assertTrue(classes.exists());
				Assert.assertTrue(log, exec.isSuccess());
			} else if (tag.equals(COMPILE)) {
				File ajcProjectClass = new File(layout.getClassesDir(),
						"org/ajc/system/AjcProject.class");
				Assert.assertTrue(ajcProjectClass.exists());
				Assert.assertTrue(log, exec.isSuccess());
			} else if (tag.equals(HELP)) {
				Assert.assertTrue(stdout, stdout.contains("hello-ajc"));
				Assert.assertTrue(log, exec.isSuccess());
			} else if (tag.equals(HELLO_AJC)) {
				Assert.assertTrue(stdout, stdout
						.contains("hello from this sample project!"));
				Assert.assertTrue(log, exec.isSuccess());
			} else if (tag.equals(CLEAN)) {
				Assert.assertTrue(stdout, stdout
						.contains("load time weaving appears to work"));
				Assert.assertTrue(log, exec.isSuccess());
			}

			layout.getLogFile().delete();
		}
	}

}
