/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.core;

import java.io.File;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.tools.ant.taskdefs.Copy;
import org.apache.tools.ant.types.DirSet;
import org.apache.tools.ant.types.FileSet;

import protoj.core.internal.AntTarget;
import protoj.core.internal.CoreProject;
import protoj.core.internal.InformationException;

/**
 * Compiles the project source Java (.java) files. Compilation of AspectJ (.aj)
 * files is also supported without any additional software. Note that the
 * {@link #compile(String, String)} method is used to fully compile the project
 * from a programmatic call or from the compile command. The
 * {@link #compileAjc(List)} and {@link #compileJavac(List)} methods are used
 * usually from the command line as an alternative to calling the initial javac
 * in order to compile the core bootstrapping part of this project.
 * 
 * 
 * @author Ashley Williams
 * 
 */
public final class CompileFeature {

	/**
	 * The project delegate pojo.
	 */
	private final CoreProject parent;

	/**
	 * The callback object used to provide extra configuration.
	 */
	private ArgRunnable<CompileFeature> config;

	/**
	 * See {@link #getAjcCompileTask()}.
	 */
	private AspectJCompileTask ajcCompileTask;

	/**
	 * See {@link #isAspectCompiler()}.
	 */
	private boolean aspectCompiler;

	/**
	 * See {@link #getJavacCompileTask()}.
	 */
	private JavacCompileTask javacCompileTask;

	/**
	 * See the getters corresponding to the parameters here.
	 * 
	 * @param parent
	 */
	public CompileFeature(CoreProject parent) {
		this.parent = parent;
		this.aspectCompiler = false;
	}

	/**
	 * See the getters corresponding to the parameters here.
	 * 
	 * @param aspectCompiler
	 * @param config
	 */
	public void initConfig(boolean aspectCompiler,
			ArgRunnable<CompileFeature> config) {
		this.aspectCompiler = aspectCompiler;
		this.config = config;
	}

	/**
	 * The ant wrapper used to carry out aspectj compilation.
	 * 
	 * @return
	 */
	public AspectJCompileTask getAjcCompileTask() {
		return ajcCompileTask;
	}

	/**
	 * The ant wrapper used to carry out javac compilation.
	 * 
	 * @return
	 */
	public JavacCompileTask getJavacCompileTask() {
		return javacCompileTask;
	}

	/**
	 * Compiles the source code based on the specified ant patterns.
	 * 
	 * @param includes
	 *            can be null to include all
	 * @param excludes
	 *            can be null to exclude nothing
	 */
	public void compile(String includes, String excludes) {
		// use if statement, don't need polymorphism
		if (aspectCompiler) {
			compileAjc(includes, excludes);
		} else {
			compileJavac(includes, excludes);
		}
	}

	/**
	 * Invokes the aspectj compiler.
	 * 
	 * @param includes
	 * @param excludes
	 */
	public void compileAjc(String includes, String excludes) {
		copyResources();
		ajcCompileTask = createAjcTask();
		DirSet dirSet = ajcCompileTask.getDirSet();
		if (includes != null) {
			dirSet.setIncludes(includes);
		}
		if (excludes != null) {
			dirSet.setExcludes(excludes);
		}
		if (config != null) {
			config.run(this);
		}
		ajcCompileTask.execute();
		if (!ajcCompileTask.isSuccess()) {
			throw new InformationException(
					"there was a problem compiling the source code");
		}
	}

	/**
	 * Invokes the javac compiler.
	 * 
	 * @param includes
	 * @param excludes
	 */
	public void compileJavac(String includes, String excludes) {
		copyResources();
		javacCompileTask = createJavacTask();
		DirSet dirSet = javacCompileTask.getDirSet();
		if (includes != null) {
			dirSet.setIncludes(includes);
		}
		if (excludes != null) {
			dirSet.setExcludes(excludes);
		}
		if (config != null) {
			config.run(this);
		}
		javacCompileTask.execute();
		if (!javacCompileTask.isSuccess()) {
			throw new InformationException(
					"there was a problem compiling the source code");
		}
	}

	/**
	 * Compiles a set of files in the source directory using the javac compiler.
	 * The options parameter contains name value pairs to be applied to the
	 * underlying ant task by beans introspection, so that the compiler can
	 * easily be configured from the command line.
	 * <p>
	 * The bean being configured is the JavacCompileTask instance. For example a
	 * string with the content "compileTask.source=1.5" is equivalent to calling
	 * <code>JavacCompileTask.getCompileTask().setSource("1.5")</code>.
	 * <p>
	 * If the source directory doesn't exist then this method does nothing.
	 * 
	 * @param options
	 */
	public void compileJavac(List<String> options) {
		File srcDir = parent.getLayout().getSrcDir();
		if (srcDir.exists()) {
			parent.getLogger().info("applying javac options:");
			JavacCompileTask task = createJavacTask();
			assignProperties(options, task);
			copyResources();
			task.execute();
			if (!task.isSuccess()) {
				throw new InformationException(
						"there was a problem compiling the source code");
			}
		}
	}

	/**
	 * Aspectj compiling equivalent of {@link #compileJavac(List)}.
	 * 
	 * @param options
	 */
	public void compileAjc(List<String> options) {
		File srcDir = parent.getLayout().getSrcDir();
		if (srcDir.exists()) {
			parent.getLogger().info("applying ajc options:");
			AspectJCompileTask task = createAjcTask();
			assignProperties(options, task);
			copyResources();
			task.execute();
			if (!task.isSuccess()) {
				throw new InformationException(
						"there was a problem compiling the source code");
			}
		}
	}

	/**
	 * Assigns the name=value pairs in the opts list to the given task using
	 * beans utils.
	 * 
	 * @param opts
	 * @param task
	 */
	private void assignProperties(List<String> opts, Object task) {
		for (String opt : opts) {
			String[] property = opt.split("=");
			String name = property[0];
			String value = property[1];
			parent.getLogger().info(String.format("%s=%s", name, value));
			PropertyUtils.setProperty(task, name, value);
		}
	}

	/**
	 * True if the ajc compiler is configured, false if the javac compiler is
	 * configured.
	 * 
	 * @return
	 */
	public boolean isAspectCompiler() {
		return aspectCompiler;
	}

	/**
	 * Creates the ajc compile implementation instance.
	 * 
	 * @return
	 */
	public AspectJCompileTask createAjcTask() {
		ProjectLayout layout = parent.getLayout();
		AspectJCompileTask task = new AspectJCompileTask(layout.getJavaDir(),
				layout.getClassesDir(), layout.getClasspathConfig());
		task.initLogging(layout.getLogFile());
		task.getCompileTask().setFailonerror(true);
		return task;
	}

	/**
	 * Creates the javac compile implementation instance.
	 * 
	 * @return
	 */
	public JavacCompileTask createJavacTask() {
		ProjectLayout layout = parent.getLayout();
		JavacCompileTask task = new JavacCompileTask(layout.getJavaDir(),
				layout.getClassesDir(), parent.getLayout().getClasspathConfig());
		task.initLogging(layout.getLogFile());
		return task;
	}

	/**
	 * Copies the files in the resources directory into the classes directory.
	 */
	public void copyResources() {
		AntTarget target = new AntTarget("compile-feature-copy");
		ProjectLayout layout = parent.getLayout();
		layout.getClassesDir().mkdirs();
		Copy copy = new Copy();
		target.addTask(copy);
		copy.setTaskName("copy");
		copy.setTodir(layout.getClassesDir());
		FileSet resourceFiles = new FileSet();
		resourceFiles.setDir(layout.getResourcesDir());
		copy.addFileset(resourceFiles);
		target.execute();
	}

}
