/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.core;

import java.io.File;

import org.apache.log4j.Appender;
import org.apache.log4j.ConsoleAppender;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.apache.log4j.PatternLayout;
import org.apache.log4j.RollingFileAppender;
import org.apache.log4j.varia.NullAppender;
import org.apache.tools.ant.listener.Log4jListener;

import protoj.core.internal.CoreProject;

/**
 * Log4j wrapper class for protoj logging. Two common appenders are supported
 * via the {@link #setConsoleAppenderEnabled} and
 * {@link #setFileAppenderEnabled} methods - simply call none, one or both
 * methods as required. The resulting appenders that get created are assigned a
 * default pattern.
 * <p>
 * This class uses a protoj rather than the global root logger so that there is
 * no conflict with other libraries in the same jvm that are also using log4j.
 * 
 * @author Ashley Williams
 * 
 *         BUG: ajc task logging doesn't work unless I add appender to root
 *         logger don't want to though because that enables appenders for other
 *         libraries also
 */
public final class ProtoLogger {
	/**
	 * See {@link #getConsolePattern()}.
	 */
	private String consolePattern = "%m%n";

	/**
	 * See {@link #getFilePattern()}.
	 */
	private String filePattern = "%d{HH:mm:ss,SSS} [%t] %m%n";

	/**
	 * See {@link #getProtoLogger()}.
	 */
	private Logger protoLogger;

	/**
	 * See {@link #getAntLogger()}.
	 */
	private Logger antLogger;

	/**
	 * Creates an instance with the owning project. After this constructor
	 * returns a file appender will be active.
	 * 
	 * @param parent
	 */
	public ProtoLogger(CoreProject parent) {
		File logFile = parent.getLayout().getLogFile();
		this.protoLogger = Logger.getLogger(logFile.getAbsolutePath());
		this.antLogger = Logger.getLogger(Log4jListener.LOG_ANT);
		boolean isEmpty = !Logger.getRootLogger().getAllAppenders()
				.hasMoreElements();
		if (isEmpty) {
			Logger.getRootLogger().addAppender(new NullAppender());
		}
		setFileAppenderEnabled(true, logFile.getAbsolutePath());
	}

	/**
	 * Adds or removes a new console consoleAppender to the log4j loggers. When
	 * adding, if an appender already exists with the same name then nothing
	 * will happen. When removing, if no appender exists with the same name then
	 * nothing will happen.
	 * 
	 * @param enable
	 *            true to create and add a new appender, false to remove any
	 *            existing appender
	 * @return
	 */
	public ConsoleAppender setConsoleAppenderEnabled(boolean enable) {
		String name = "proto.consoleAppender";
		Appender appender = Logger.getRootLogger().getAppender(name);
		boolean appenderExists = appender != null;

		if (enable) {
			if (!appenderExists) {
				appender = new ConsoleAppender(new PatternLayout(
						getConsolePattern()), ConsoleAppender.SYSTEM_OUT);
				appender.setName(name);
				// protoLogger.addAppender(appender);
				// antLogger.addAppender(appender);
				Logger.getRootLogger().addAppender(appender);
			}
		} else {
			if (appenderExists) {
				// protoLogger.removeAppender(appender);
				// antLogger.removeAppender(appender);
				Logger.getRootLogger().removeAppender(appender);
			}
		}

		return (ConsoleAppender) protoLogger.getAppender(name);
	}

	/**
	 * Adds or removes a new rolling file appender to the log4j loggers. When
	 * adding, if an appender already exists with the same name then nothing
	 * will happen. When removing, if no appender exists with the same name then
	 * nothing will happen.
	 * 
	 * @param enable
	 *            true to create and add a new appender, false to remove any
	 *            existing appender
	 * @param logFile
	 *            the log file to be written to, also used in the name of the
	 *            appender
	 * 
	 * @return
	 */
	public RollingFileAppender setFileAppenderEnabled(boolean enable,
			String logFile) {
		String name = "proto.fileAppender";
		Appender appender = Logger.getRootLogger().getAppender(name);
		boolean appenderExists = appender != null;

		if (enable) {
			if (!appenderExists) {
				appender = new RollingFileAppender(new PatternLayout(
						getFilePattern()), logFile);
				appender.setName(name);
				// protoLogger.addAppender(appender);
				// antLogger.addAppender(appender);
				Logger.getRootLogger().addAppender(appender);
			}
		} else {
			if (appenderExists) {
				// protoLogger.removeAppender(appender);
				// antLogger.removeAppender(appender);
				Logger.getRootLogger().removeAppender(appender);
			}
		}
		return (RollingFileAppender) appender;
	}

	public String getConsolePattern() {
		return consolePattern;
	}

	public String getFilePattern() {
		return filePattern;
	}

	public Logger getProtoLogger() {
		return protoLogger;
	}

	public Logger getAntLogger() {
		return antLogger;
	}

	public void setLevel(Level level) {
		// protoLogger.setLevel(level);
		// antLogger.setLevel(level);
		Logger.getRootLogger().setLevel(level);
	}

}
