/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.core.internal;

import java.io.File;

import org.apache.log4j.Logger;

import protoj.core.CommandStore;
import protoj.core.CompileFeature;
import protoj.core.DispatchFeature;
import protoj.core.InstructionChain;
import protoj.core.ProjectLayout;
import protoj.core.PropertyStore;
import protoj.core.ProtoLogger;
import protoj.core.ResourceFeature;
import protoj.core.RetrieveFeature;
import protoj.core.StandardProperties;

/**
 * Provides a minimal amount of protoj infrastructure through delegate helpers
 * so that the protoj project can be built in the first place. Projects should
 * instead use the <code>StandardProject</code> class as a starting point, which
 * in turn delegates methods from this class to implement the requested
 * features. Please see that class for documentation of the identically named
 * methods.
 * 
 * @author Ashley Williams
 * 
 */
public final class CoreProject {
	private PropertyStore propertyStore;

	private StandardProperties properties;

	private CommandStore commandStore;

	private CompileCommand compileCommand;

	private JavacCommand javacCommand;

	private AjcCommand ajcCommand;

	private RetrieveCommand retrieveCommand;

	private CleanCommand cleanCommand;

	private ProjectLayout layout;

	private ProtoLogger protoLogger;

	private InstructionChain instructionChain;

	private RetrieveFeature retrieveFeature;

	private DispatchFeature dispatchFeature;

	private CompileFeature compileFeature;

	private ResourceFeature resourceFeature;

	public CoreProject(String[] args) {
		this(new InstructionChain(args));
	}

	public CoreProject(File rootDir, String scriptName) {
		this(new InstructionChain(rootDir, scriptName));
	}

	private CoreProject(InstructionChain instructionChain) {
		this.instructionChain = instructionChain;
		this.layout = new ProjectLayout(instructionChain.getRootDir(),
				instructionChain.getScriptName());
		this.protoLogger = new ProtoLogger(this);
		this.commandStore = new CommandStore(this);
		this.propertyStore = new PropertyStore(this);
		this.properties = new StandardProperties(this);
		this.compileCommand = new CompileCommand(this);
		this.javacCommand = new JavacCommand(this);
		this.ajcCommand = new AjcCommand(this);
		this.retrieveCommand = new RetrieveCommand(this);
		this.cleanCommand = new CleanCommand(this);
		this.dispatchFeature = new DispatchFeature(this);
		this.resourceFeature = new ResourceFeature(this);
		this.compileFeature = new CompileFeature(this);
		this.retrieveFeature = new RetrieveFeature(this);
	}

	public ResourceFeature getResourceFeature() {
		return resourceFeature;
	}

	public InstructionChain getInstructionChain() {
		return instructionChain;
	}

	public DispatchFeature getDispatchFeature() {
		return dispatchFeature;
	}

	public ProjectLayout getLayout() {
		return layout;
	}

	public ProtoLogger getProtoLogger() {
		return protoLogger;
	}

	public Logger getLogger() {
		return protoLogger.getProtoLogger();
	}

	public CommandStore getCommandStore() {
		return commandStore;
	}

	public PropertyStore getPropertyStore() {
		return propertyStore;
	}

	public StandardProperties getProperties() {
		return properties;
	}

	public JavacCommand getJavacCommand() {
		return javacCommand;
	}

	public AjcCommand getAjcCommand() {
		return ajcCommand;
	}

	public CompileCommand getCompileCommand() {
		return compileCommand;
	}

	public RetrieveCommand getRetrieveCommand() {
		return retrieveCommand;
	}

	public CleanCommand getCleanCommand() {
		return cleanCommand;
	}

	public RetrieveFeature getRetrieveFeature() {
		return retrieveFeature;
	}

	public CompileFeature getCompileFeature() {
		return compileFeature;
	}
}
