/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.lang;

import protoj.core.ArgRunnable;
import protoj.core.ProjectLayout;

/**
 * Provides support for the creation of source, javadoc, classes jar files and
 * project tar file. This class is really just an aggregate of implementation
 * objects so see the {@link SourceArchive}, {@link JavadocArchive},
 * {@link ClassesArchive} and {@link ProjectArchive} classes for more
 * information.
 * 
 * @author Ashley Williams
 * 
 */
public final class ArchiveFeature {
	/**
	 * See {@link #getClassesArchive()}.
	 */
	private ClassesArchive classesArchive;

	/**
	 * See {@link #getSourceArchive()}.
	 */
	private SourceArchive sourceArchive;

	/**
	 * See {@link #getJavadocArchive()}.
	 */
	private JavadocArchive javadocArchive;

	/**
	 * See {@link #getProjectArchive()}.
	 */
	private ProjectArchive projectArchive;

	/**
	 * See {@link #getProject()}.
	 */
	private final StandardProject project;

	private ProjectLayout layout;

	/**
	 * Enables creation of source, javadoc and classes jar files.
	 * 
	 * @param parent
	 */
	public ArchiveFeature(StandardProject parent) {
		this.project = parent;
		this.layout = parent.getLayout();
		this.classesArchive = new ClassesArchive(this);
		this.sourceArchive = new SourceArchive(this);
		this.javadocArchive = new JavadocArchive(this);
	}

	/**
	 * This method must be called if support for creating a compressed tar file
	 * of the project is required. If no additional customization is required,
	 * simply provide a null config argument. See {@link ProjectArchive} for an
	 * example of a config argument.
	 * 
	 * @param name
	 *            the name of the tar file without any extension
	 * @param prefix
	 *            used as the name of the root directory when the tarfile is
	 *            extracted
	 * @param userName
	 *            the user name for the tar entry, can be null
	 * @param group
	 *            the group for the tar entry, can be null
	 * @param config
	 *            extra configuration, can be null
	 */
	public void initProjectArchive(String name, String prefix, String userName,
			String group, ArgRunnable<ProjectArchive> config) {
		this.projectArchive = new ProjectArchive(this, name, prefix, userName,
				group, config);
	}

	/**
	 * Convenient wrapper when just the default tar is required with no
	 * customization and applying tar name suffixes as necessary. See
	 * {@link #initProjectArchive(String, String, String, String, ArgRunnable)}.
	 * 
	 * @param name
	 * @param prefix
	 */
	public void initProjectArchive(String name, String prefix) {
		initProjectArchive(name, prefix, null, null, null);
	}

	/**
	 * Convenience method that schedules the creation of a single jar file with
	 * the given name. See
	 * {@link ClassesArchive#addArchive(String, String, String, String, ArgRunnable)}
	 * .
	 * 
	 * @param name
	 */
	public void addClasses(String name) {
		getClassesArchive().addArchive(name, null, null, null, null);
	}

	/**
	 * Convenience method that schedules the creation of a single javadoc
	 * archive file with the given name and memory. See
	 * {@link JavadocArchive#addArchive(String, String, String, String, String, ArgRunnable)}
	 * . .
	 * 
	 * @param name
	 */
	public void addJavadoc(String name, String memory) {
		getJavadocArchive().addArchive(name, null, null, null, memory, null);
	}

	/**
	 * Convenience method that schedules the creation of a single sources
	 * archive file with the given name and memory. See
	 * {@link JavadocArchive#addArchive(String, String, String, String, String, ArgRunnable)}
	 * . .
	 * 
	 * @param name
	 */
	public void addSources(String name) {
		getSourceArchive().addArchive(name, null, null, null, null);
	}

	/**
	 * Returns whether or not the given jar name represents a valid classes jar
	 * file. This is a jar file that doesn't contain the strings
	 * {@link ProjectLayout#getSourcePostfix()},
	 * {@link ProjectLayout#getJavadocPostfix()} or
	 * {@link ProjectLayout#getSrcPostfix()}.
	 * 
	 * @param name
	 * @return
	 */
	public boolean isClassesJar(String name) {
		return name.endsWith(".jar") && !isSourcesJar(name)
				&& !isJavadocJar(name);
	}

	/**
	 * Returns whether or not the given jar name represents a valid javadoc jar
	 * file. This is a jar file that contain the string
	 * {@link ProjectLayout#getJavadocPostfix()}.
	 * 
	 * @param name
	 * @return
	 */
	public boolean isJavadocJar(String name) {
		return name.endsWith(".jar")
				&& name.contains(layout.getJavadocPostfix());
	}

	/**
	 * Returns whether or not the given jar name represents a valid sources jar
	 * file. This is a jar file that contains either of the strings
	 * {@link ProjectLayout#getSourcePostfix()} or
	 * {@link ProjectLayout#getSrcPostfix()}.
	 * 
	 * @param name
	 * @return
	 */
	public boolean isSourcesJar(String name) {
		return name.endsWith(".jar")
				&& (name.contains(layout.getSourcePostfix()) || name
						.contains(layout.getSrcPostfix()));
	}

	/**
	 * Use this instance to configure the creation of java archives from the
	 * classes directory. See {@link ClassesArchive}.
	 * 
	 * @return
	 */
	public ClassesArchive getClassesArchive() {
		return classesArchive;
	}

	/**
	 * Use this instance to configure the creation of java archives from the
	 * source directory. See {@link SourceArchive}.
	 * 
	 * @return
	 */
	public SourceArchive getSourceArchive() {
		return sourceArchive;
	}

	/**
	 * Use this instance to configure the creation of java archives from the
	 * javadoc directory. See {@link JavadocArchive}.
	 * 
	 * @return
	 */
	public JavadocArchive getJavadocArchive() {
		return javadocArchive;
	}

	/**
	 * Use this instance to configure the creation of a compressed tar file for
	 * the entire application. See {@link ProjectArchive}.
	 * 
	 * @return
	 */
	public ProjectArchive getProjectArchive() {
		return projectArchive;
	}

	/**
	 * The parent project for this feature.
	 * 
	 * @return
	 */
	public StandardProject getProject() {
		return project;
	}

	/**
	 * The parent layout for convenient access.
	 * 
	 * @return
	 */
	public ProjectLayout getLayout() {
		return layout;
	}

}
