/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.lang.internal;

import joptsimple.OptionSet;
import joptsimple.OptionSpec;
import protoj.core.Command;
import protoj.core.CommandStore;
import protoj.lang.PublishFeature;
import protoj.lang.StandardProject;

/**
 * Publishes project artifacts to a maven repository by performing a maven
 * deploy. The repository url and additional configuration must be performed
 * in-code. The following options are supported:
 * <ol>
 * <li>-artifact: the name of the artifact to be published. The artifacts
 * directory is checked for classes, source and javadoc containing artifacts
 * that match the name. If this option isn't specified then all discovered
 * artifacts are published.</li>
 * <li>-user: used if the publish command has been configured for a remote
 * protocol such as scp.</li>
 * <li>-key: the path to the private key for scp key-based authentication. If
 * not specified then the default key location is used if it exists.</li>
 * <li>-password: the password for scp transport if specified.</li>
 * <li>-passphrase: the passphrase for scp key-based authentication. Can be left
 * out if there is no passphrase for the given key.</li>
 * </ol>
 * 
 * Example: publish Example: "publish -user bob" Example:
 * "publish -user bob -key ~/.ssh/myprivatekey"
 * 
 * @author Ashley Williams
 * 
 */
public final class PublishCommand {

	public final class Body implements Runnable {
		public void run() {
			OptionSet options = getDelegate().getOptions();

			String user = options.valueOf(userOption);
			String key = options.valueOf(keyOption);
			String password = options.valueOf(passwordOption);
			String passphrase = options.valueOf(passphraseOption);
			PublishFeature feature = project.getPublishFeature();
			if (options.has(getArtifactOption())) {
				String artifact = options.valueOf(getArtifactOption());
				feature.deploy(artifact, user, key, password, passphrase);
			} else {
				feature.deployAll(user, key, password, passphrase);
			}

		}
	}

	/**
	 * Provides the basic command functionality.
	 */
	private Command delegate;
	private OptionSpec<String> artifactOption;
	private StandardProject project;
	private OptionSpec<String> userOption;
	private OptionSpec<String> keyOption;
	private OptionSpec<String> passwordOption;
	private OptionSpec<String> passphraseOption;

	public PublishCommand(StandardProject project) {
		this.project = project;
		CommandStore store = project.getCommandStore();
		String scriptName = project.getLayout().getScriptName();

		StringBuilder description = new StringBuilder();
		description
				.append("\nPublishes project artifacts to a maven repository by performing a maven");
		description
				.append("\ndeploy. The repository url and additional configuration must be performed");
		description.append("\nin-code. The following options are supported:");
		description.append("\n");
		description
				.append("\n   1. -artifact: the name of the artifact to be published. The artifacts");
		description
				.append("\n      directory is checked for classes, source and javadoc containing artifacts");
		description
				.append("\n      that match the name. If this option isn't specified then all discovered");
		description.append("\n      artifacts are published.");
		description.append("\n");
		description
				.append("\n   2. -user: used if the publish command has been configured for a remote");
		description.append("\n      protocol such as scp.");
		description.append("\n");
		description
				.append("\n   3. -key: the path to the private key for scp key-based authentication. If");
		description
				.append("\n      not specified then the default key location is used if it exists.");
		description.append("\n");
		description
				.append("\n   4. -password: the password for scp transport if specified.");
		description.append("\n");
		description
				.append("\n   5. -passphrase: the passphrase for scp key-based authentication. Can be left");
		description
				.append("\n      out if there is no passphrase for the given key.");
		description.append("\n");
		description.append(String.format("\nExample: jonny$ ./%s publish",
				scriptName));
		description.append(String.format(
				"\nExample: jonny$ ./%s \"publish -user bob\"", scriptName));
		description
				.append(String
						.format(
								"\nExample: jonny$ ./%s \"publish -user bob -key ~/.ssh/myprivatekey\"",
								scriptName));

		delegate = store.addCommand("publish", description.toString(), "16m",
				new Body());
		delegate.initAliases("deploy");
		artifactOption = delegate.getParser().accepts("artifact")
				.withRequiredArg();
		userOption = delegate.getParser().accepts("user").withRequiredArg();
		keyOption = delegate.getParser().accepts("key").withRequiredArg();
		passwordOption = delegate.getParser().accepts("password")
				.withRequiredArg();
		passphraseOption = delegate.getParser().accepts("passphrase")
				.withRequiredArg();
	}

	public OptionSpec<String> getArtifactOption() {
		return artifactOption;
	}

	public Command getDelegate() {
		return delegate;
	}
}
