/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.lang.internal;

import joptsimple.OptionParser;
import joptsimple.OptionSet;
import joptsimple.OptionSpec;
import joptsimple.OptionSpecBuilder;
import protoj.core.Command;
import protoj.core.CommandStore;

/**
 * The command responsible for performing a release of protoj to the google code
 * website and sourceforge maven repository.
 * 
 * @author Ashley Williams
 * 
 */
public final class ReleaseCommand {
	private final class Body implements Runnable {
		private final ProtoProject project;

		private Body(ProtoProject project) {
			this.project = project;
		}

		public void run() {
			OptionSet options = delegate.getOptions();
			if (!options.has(gcUserOption)) {
				String message = "incorrect arguments: the release command requires a googlecode username";
				delegate.throwBadOptionsException(message);
			}
			if (!options.has(gcPasswordOption)) {
				String message = "incorrect arguments: the release command requires a googlecode password";
				delegate.throwBadOptionsException(message);
			}
			if (!options.has(sfUserOption)) {
				String message = "incorrect arguments: the release command requires a sourceforge username";
				delegate.throwBadOptionsException(message);
			}

			String gcUser = options.valueOf(gcUserOption);
			String gcPassword = options.valueOf(gcPasswordOption);
			String sfUser = options.valueOf(sfUserOption);

			if (options.has(gcOnlyOption)) {
				project.getReleaseFeature().uploadToGoogleCode(gcUser,
						gcPassword);
			} else if (options.has(sfOnlyOption)) {
				project.getReleaseFeature().uploadToMaven(sfUser);
			} else {
				project.getReleaseFeature().release(gcUser, gcPassword, sfUser);
			}
		}
	}

	/**
	 * Provides the basic command functionality.
	 */
	private Command delegate;
	private OptionSpec<String> gcUserOption;
	private OptionSpec<String> gcPasswordOption;
	private OptionSpec<String> sfUserOption;
	private OptionSpecBuilder gcOnlyOption;
	private OptionSpecBuilder sfOnlyOption;

	public ReleaseCommand(final ProtoProject project, String mainClass) {
		CommandStore store = project.getDelegate().getCommandStore();
		delegate = store
				.addCommand(
						"release",
						"Performs all the steps on a compiled project in\norder to release protoj to the host google code website.",
						"64m", new Body(project));
		OptionParser parser = delegate.getParser();
		gcUserOption = parser.accepts("gcuser").withRequiredArg().ofType(
				String.class);
		gcPasswordOption = parser.accepts("gcpassword").withRequiredArg()
				.ofType(String.class);
		sfUserOption = parser.accepts("sfuser").withRequiredArg().ofType(
				String.class);
		gcOnlyOption = parser.accepts("gconly");
		sfOnlyOption = parser.accepts("sfonly");
	}

	public Command getDelegate() {
		return delegate;
	}

}
