/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.lang.internal;

import java.io.File;
import java.io.FilenameFilter;

import org.apache.commons.io.FileUtils;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.taskdefs.GUnzip;
import org.apache.tools.ant.taskdefs.Untar;

import protoj.core.ProjectLayout;
import protoj.core.PropertyInfo;
import protoj.core.internal.AntTarget;
import protoj.core.internal.InformationException;
import protoj.lang.ArchiveEntry;
import protoj.lang.ArchiveFeature;
import protoj.lang.ClassesArchive;
import protoj.lang.ProjectArchive;
import protoj.lang.PublishFeature;
import protoj.lang.StandardProject;

/**
 * Responsible for kicking off a release of the ProtoJ project. The release
 * culminates in a deployment of the java artifacts to the maven repository at
 * source forge and the upload of the project tar archive to googlecode.
 * <p>
 * The only prerequisite for invoking the
 * {@link #release(String, String, String)} method is that the protoj code is
 * compiled.
 * 
 * @author Ashley Williams
 * 
 */
public final class ReleaseFeature {
	private final ProtoProject project;

	public ReleaseFeature(ProtoProject project) {
		this.project = project;
	}

	/**
	 * Carries out the steps to release protoj starting from just the compiled
	 * code.
	 * 
	 * @param gcUserName
	 *            the username for uploading to the googlecode project
	 * @param gcPassword
	 *            the password for uploading to the googlecode project
	 * @param sfUserName
	 *            the username for uploading to the sourceforge maven repository
	 * 
	 */
	public void release(String gcUserName, String gcPassword, String sfUserName) {
		showSummary();
		extractSite();
		createArchives();
		verifyProjectArchive();
		uploadToMaven(sfUserName);
		uploadToGoogleCode(gcUserName, gcPassword);
	}

	/**
	 * Just extracts the googlecode website files from the jar and copies to the
	 * checked out wiki directory as a sibling of the protoj project root dir.
	 */
	private void extractSite() {
		ProjectLayout layout = project.getDelegate().getLayout();
		File wikiDir = new File(layout.getRootDir().getParentFile(), "wiki");
		if (!wikiDir.exists()) {
			String message = "checked out wiki directory is required: "
					+ wikiDir.getAbsolutePath();
			throw new RuntimeException(message);
		}
		File dir = project.extractSite();
		File srcDir = new File(dir, "protoj/site/wiki");
		FileUtils.copyDirectory(srcDir, wikiDir);
	}

	/**
	 * Publishes the archives to the sourceforge maven repository.
	 * 
	 * @param sfUserName
	 */
	public void uploadToMaven(String sfUserName) {
		PublishFeature publishFeature = project.getDelegate()
				.getPublishFeature();
		publishFeature.deployAll(sfUserName, null, null, null);
	}

	/**
	 * Creates the tar and jar files that are to be uploaded to googlecode.
	 */
	private void createArchives() {
		StandardProject delegate = project.getDelegate();
		ArchiveFeature archive = delegate.getArchiveFeature();
		archive.getClassesArchive().createArchives();
		archive.getJavadocArchive().createArchives();
		archive.getSourceArchive().createArchives();
		archive.getProjectArchive().createArchive(false, true);
	}

	/**
	 * Verifies the project tar, by extracting and executing the compile and
	 * junit tests inside. These are the steps that resemble a proper download
	 * of the uploaded tar file. Also looks for additional protoj related
	 * project information.
	 */
	private void verifyProjectArchive() {
		project.getDelegate().getVerifyTarFeature().verifyTar();

		final ProtoProject extractedProject = createExtractedProject();
		final StandardProject delegate = extractedProject.getDelegate();
		final ProjectLayout extractedLayout = delegate.getLayout();
		File extractedRootDir = extractedLayout.getRootDir();

		File readmeFile = new File(extractedRootDir, "README.txt");
		if (!readmeFile.exists()) {
			String reason = "can't find file " + readmeFile.getAbsolutePath();
			throw new InformationException(reason);
		}

		File licenseFile = new File(extractedRootDir, "LICENSE.txt");
		if (!licenseFile.exists()) {
			String reason = "can't find file " + licenseFile.getAbsolutePath();
			throw new InformationException(reason);
		}

		File noticeFile = new File(extractedRootDir, "NOTICE.txt");
		if (!noticeFile.exists()) {
			String reason = "can't find file " + noticeFile.getAbsolutePath();
			throw new InformationException(reason);
		}

		File exampleDir = new File(extractedRootDir, "example");
		if (!exampleDir.exists()) {
			String reason = "can't find directory "
					+ exampleDir.getAbsolutePath();
			throw new InformationException(reason);
		}

		File javadocDir = new File(extractedLayout.getDocsDir(), "javadoc");
		if (!exampleDir.exists()) {
			String reason = "can't find javadoc dir "
					+ javadocDir.getAbsolutePath();
			throw new InformationException(reason);
		}

		extractedLayout.getConfDir().list(new FilenameFilter() {
			public boolean accept(File dir, String name) {
				if (name.contains(".properties")) {
					String reason = "no properties files allowed in conf directory: "
							+ name;
					throw new InformationException(reason);
				}
				return false;
			}
		});

		extractedLayout.getLibDir().list(new FilenameFilter() {
			public boolean accept(File dir, String name) {
				ArchiveFeature archive = delegate.getArchiveFeature();
				boolean sourcesJar = archive.isSourcesJar(name);
				if (sourcesJar) {
					String reason = "no source or javadoc jars allowed in lib directory: "
							+ name;
					throw new InformationException(reason);
				}
				return false;
			}

		});
	}

	/**
	 * Uploads the project archive, the jar with dependencies and the jar with
	 * no dependencies to google code.
	 * 
	 * @param gcUserName
	 * @param gcPassword
	 */
	public void uploadToGoogleCode(String gcUserName, String gcPassword) {
		uploadArtifact(project.getJarName(),
				"Merged jar file with maximum dependencies", gcUserName,
				gcPassword, "Type-Archive, OpSys-All");
		uploadArtifact(project.getNoDepJarName(),
				"Merged executable jar file with minimum dependencies",
				gcUserName, gcPassword, "Type-Archive, OpSys-All, Featured");
		// seem to have hit a file size limit, do by hand for now
		// uploadProjectArchive(gcUserName, gcPassword);
	}

	/**
	 * Uploads the project tar archive to the googlecode website.
	 * 
	 * @param userName
	 * @param password
	 */
	private void uploadProjectArchive(String userName, String password) {
		StandardProject delegate = project.getDelegate();
		String summary = "Build from source with: ./protoj.sh compile archive";
		String labels = "Type-Source, OpSys-All";
		String googleName = String.format("%s-src.tar.gz", project
				.getProjectName());
		delegate.getUploadGoogleCodeFeature().uploadTar(googleName, labels,
				summary, userName, password);
	}

	/**
	 * Uploads the artifact with the given name to the googlecode website.
	 * 
	 * @param name
	 * @param description
	 * @param userName
	 * @param password
	 * @param labels
	 */
	private void uploadArtifact(String name, String description,
			String userName, String password, String labels) {
		StandardProject delegate = project.getDelegate();
		String summary = description;
		ArchiveFeature archive = delegate.getArchiveFeature();
		ClassesArchive classes = archive.getClassesArchive();
		ArchiveEntry<ClassesArchive> entry = classes.getEntry(name)
				.getArchiveEntry();
		String googleName = String.format("%s-%s.jar", entry.getName(), project
				.getVersionNumber());
		File artifact = entry.getArtifact();
		delegate.getUploadGoogleCodeFeature().uploadArtifact(artifact,
				googleName, labels, summary, userName, password);
	}

	/**
	 * The release takes a long time so it's worth knowing whether or not an
	 * upload to google code will take place at the end of it.
	 */
	private void showSummary() {
		StandardProject delegate = project.getDelegate();
		PropertyInfo gcskip = delegate.getProperties().getGcskipProperty();
		StringBuilder builder = new StringBuilder();
		if (gcskip.getBooleanValue() == true) {
			builder
					.append("ProtoJ won't be uploaded to google code during this release.");
		} else {
			builder
					.append("ProtoJ will be uploaded to google code during this release.");
		}
		builder.append(" See the help for ");
		builder.append(gcskip.getKey());
		builder.append(" for more information.");
		delegate.getLogger().info(builder.toString());
	}

	/**
	 * Extracts the project archive into the working directory.
	 * 
	 * @return an instance representing the extracted project
	 */
	private ProtoProject createExtractedProject() {
		StandardProject delegate = project.getDelegate();

		ProjectLayout layout = delegate.getLayout();
		ProjectArchive archive = delegate.getArchiveFeature()
				.getProjectArchive();
		File compressedFile = new File(archive.getArchivePath());

		// create the parent ant target
		AntTarget target = new AntTarget("release-extract-archive");
		target.initLogging(layout.getLogFile(), Project.MSG_INFO);

		// applies the gunzip
		GUnzip gunzip = new GUnzip();
		gunzip.setTaskName("release-gunzip");
		target.addTask(gunzip);
		gunzip.setSrc(compressedFile);

		// applies the untar
		Untar untar = new Untar();
		untar.setTaskName("release-untar");
		target.addTask(untar);
		File uncompressedFile = new File(archive.getUncompressedPath());
		untar.setSrc(uncompressedFile);
		File workingDir = new File(layout.getTargetDir(), "release");
		workingDir.mkdir();
		untar.setDest(workingDir);

		target.execute();

		File tarDir = new File(workingDir, archive.getPrefix());
		return new ProtoProject(tarDir, layout.getScriptName());
	}

}
