/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.lang.internal;

import java.io.File;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.Manifest;

import org.junit.Assert;

/**
 * Used to test correct form of a java archive file. Standard junit assertions
 * are used if any problem is detected when the {@link #execute()} method is
 * invoked.
 * 
 * @author Ashley Williams
 * 
 */
public final class VerifyArchive {
	private final File archive;
	private String[] includedResources;
	private String[] excludedResources;
	private Manifest manifest;

	/**
	 * The file to be tested.
	 * 
	 * @param archive
	 */
	public VerifyArchive(File archive) {
		this.archive = archive;
	}

	/**
	 * Any resources to check for inclusion.
	 * 
	 * @param includedResources
	 */
	public void initIncludedResources(String... includedResources) {
		this.includedResources = includedResources;
	}

	/**
	 * Any resources to check for exclusion.
	 * 
	 * @param excludedResources
	 */
	public void initExcludedResources(String... excludedResources) {
		this.excludedResources = excludedResources;
	}

	/**
	 * Asserts that the given jar file exists and contains or does not contain
	 * resources as listed by the names parameter. The manifest is made
	 * available for custom testing, just call the {@link #getManifest()}
	 * method.
	 */
	public void execute() {
		String archivePath = archive.getAbsolutePath();
		String noArchive = "couldn't find " + archivePath;
		Assert.assertTrue(noArchive, archive.exists());
		ArrayList<String> resources = new ArrayList<String>();
		JarFile jarFile = new JarFile(archive);
		try {
			Enumeration<JarEntry> entries = jarFile.entries();
			while (entries.hasMoreElements()) {
				JarEntry entry = entries.nextElement();
				resources.add(entry.getName());
			}

			if (includedResources != null) {
				for (String resource : includedResources) {
					String noResource = String.format(
							"couldn't find resource %s in %s", resource,
							archivePath);
					Assert.assertTrue(noResource, resources.contains(resource));
				}
			}

			if (excludedResources != null) {
				for (String resource : excludedResources) {
					String message = String.format(
							"mistakenly found resource %s in %s", resource,
							archivePath);
					Assert.assertFalse(message, resources.contains(resource));
				}
			}
			manifest = jarFile.getManifest();
		} finally {
			jarFile.close();
		}
	}

	/**
	 * After the {@link #execute()} method has finished, this method can be
	 * called for further testing on the manifest, that is considered to
	 * specialized to make generic here.
	 * 
	 * @return
	 */
	public Manifest getManifest() {
		return manifest;
	}

}
