/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.lang.internal;

import protoj.core.Command;
import protoj.core.CommandStore;
import protoj.lang.StandardProject;

/**
 * The command responsible for determining the validity of the project tar file.
 * 
 * @author Ashley Williams
 * 
 */
public final class VerifyTarCommand {
	private final class Body implements Runnable {
		public void run() {
			project.getVerifyTarFeature().verifyTar();
		}
	}

	/**
	 * Provides the basic command functionality.
	 */
	private Command delegate;
	private final StandardProject project;

	public VerifyTarCommand(StandardProject project) {
		this.project = project;
		CommandStore store = project.getCommandStore();
		StringBuilder description = new StringBuilder();
		description
				.append("Verifies the correct functionality of a project tar file by expanding, then");
		description
				.append("\nbuilding and executing the unit tests inside. This can prevent the situation");
		description
				.append("\nwhere extreme care has been taken over the project source code, but correct");
		description
				.append("\nfunctionality of the subsequently released tar file is left to chance. Here");
		description.append("\nare the basic steps that are taken:");
		description
				.append("\n\n   1. The tar file is extracted into the target directory, resulting in a");
		description
				.append("\n      failure if this goes wrong for some reason.");
		description
				.append("\n\n   2. If there is no source directory then this results in a failure to avoid the");
		description
				.append("\n      situation where the caller mistakenly believes the tests to have been executed.");
		description
				.append("\n\n   3. Maximum 'rwx' permissions are applied to the extracted files, since");
		description
				.append("\n      tighter restrictions can prevent compilation from working. One consequence is");
		description
				.append("\n      that the unmodified file permissions don't get tested.");
		description
				.append("\n\n   4. The extracted project script is then executed using the compile, assemble");
		description
				.append("\n      and test commands. A failure of this execution results in command failure.");
		delegate = store.addCommand("verify-tar", description.toString(),
				"16m", new Body());
		delegate.initAliases("verify-gtar", "verify-package");
	}

	public Command getDelegate() {
		return delegate;
	}

}
