/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.lang.internal.acme;

import org.junit.Assert;

import protoj.core.ArgRunnable;
import protoj.core.ProjectLayout;
import protoj.lang.ScriptSession;
import protoj.lang.internal.ant.CommandTask;

/**
 * These assertions trigger when the scp command is specified.
 * <p>
 * On the first call there is no tar file to upload so we check for the
 * appropriate error. On the second call there is a tar file, but no destination
 * has been specified so we check for the appropriate error. On the third and
 * final call there is a tar file but the upload does not actually take place.
 * Instead for testing purposes, configuration information is logged for us to
 * check.
 * <p>
 * The reason the upload does not take place is because the -Dprotoj.scpskip
 * property is specified along with the upload command. In fact that property is
 * specified in the acme.properties file, so this also provides a minor test
 * that the configuration mechanism is working correctly.
 * 
 * @author Ashley Williams
 * 
 */
final class AssertScp implements ArgRunnable<ScriptSession> {
	public static final String OK = "ok";

	public static final String NO_TAR = "no tar";

	public static final String NO_ARGS = "no args";

	private final AcmeSession acmeSession;

	private int testIndex = 0;

	public AssertScp(AcmeSession acmeSession) {
		this.acmeSession = acmeSession;
	}

	public void run(ScriptSession projectSession) {
		ProjectLayout acmeLayout = acmeSession.getProject().getLayout();
		String log = acmeSession.getProject().getLayout().loadLog();
		String tag = projectSession.getCurrentTag();

		if (tag.equals(NO_TAR)) {
			// no tar file
			Assert.assertTrue(log, log.contains("no tar file"));
		} else if (tag.equals(NO_ARGS)) {
			// no --nokey or --key options
			Assert.assertTrue(log, log.contains("missing mandatory option"));
		} else if (tag.equals(OK)) {
			// should be fine, just check the configuration details
			CommandTask exec = projectSession.getCurrentExec();
			Assert.assertTrue(exec.getResult(), exec.isSuccess());
		}

		acmeLayout.getLogFile().delete();
		testIndex++;
	}

}