/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.core;

/**
 * Represents a property in the {@link PropertyStore}. Where there is no fully
 * defined property available in the PropertyStore each PropertyInfo instance
 * contains defaults to be used instead.
 * 
 * @author Ashley Williams
 * 
 */
public final class PropertyInfo {

	/**
	 * see {@link #getKey()}()
	 */
	private String key;

	/**
	 * see {@link #getEmptyDefault()}()
	 */
	private String emptyDefault;

	/**
	 * see {@link #getMissingDefault()}()
	 */
	private String missingDefault;

	/**
	 * See {@link #getDescription}.
	 */
	private String description;

	/**
	 * The parent property store that this property belongs to.
	 */
	private final PropertyStore store;

	/**
	 * See the corresponding property descriptions.
	 * 
	 * @param store
	 * @param key
	 * @param missingDefault
	 * @param emptyDefault
	 * @param description
	 */
	public PropertyInfo(PropertyStore store, String key, String missingDefault,
			String emptyDefault, String description) {
		this.store = store;
		this.key = key;
		this.missingDefault = missingDefault;
		this.emptyDefault = emptyDefault;
		this.description = description;
	}

	/**
	 * The property key.
	 * 
	 * @return
	 */
	public String getKey() {
		return key;
	}

	/**
	 * Provides a value for the underlying key in all cases:
	 * <ul>
	 * <li>Where a value is present, that value is used. For example:
	 * <code>-Dmyproperty=myvalue</code></li>
	 * <li>Where there is no property with the given <code>key</code>,
	 * <code>missingDefault</code> is used.</li>
	 * <li>Where there is a property with the given <code>key</code> but an
	 * empty value is provided, <code>emptyDefault</code> is used. For example:
	 * <code>-Dmyproperty</code> or <code>-Dmyproperty=</code></li>
	 * </ul>
	 * 
	 * @return
	 */
	public String getValue() {
		String value = store.getConfig().getString(key, missingDefault);
		if (value.equals("")) {
			value = emptyDefault;
		}
		return value;
	}

	/**
	 * Conveniently typed method for converting {@link #getValue()}.
	 * 
	 * @return
	 */
	public boolean getBooleanValue() {
		return Boolean.parseBoolean(getValue());
	}

	/**
	 * Conveniently typed method for converting {@link #getValue()}.
	 * 
	 * @return
	 */
	public int getIntegerValue() {
		return Integer.parseInt(getValue());
	}

	/**
	 * The default value to use when the value of a property is empty. For
	 * example: <code>-Dmyproperty</code> or <code>-Dmyproperty=</code>
	 * 
	 * @return
	 */
	public String getEmptyDefault() {
		return emptyDefault;
	}

	/**
	 * The default value to use when a property isn't specified at all.
	 * 
	 * @return
	 */
	public String getMissingDefault() {
		return missingDefault;
	}

	/**
	 * A description useful for help text.
	 * 
	 * @return
	 */
	public String getDescription() {
		return description;
	}

	/**
	 * Creates a command line equivalent of this property with the specified
	 * value. Convenient for writing test cases.
	 * 
	 * @param value
	 *            can be null for a property with no value
	 * @return
	 */
	public String createJvmArg(String value) {
		StringBuilder builder = new StringBuilder();
		builder.append("-D");
		builder.append(getKey());
		if (value != null) {
			builder.append("=");
			builder.append(value);
		}
		return builder.toString();
	}
}
