/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.lang;

import java.util.Collection;

import protoj.core.Command;
import protoj.core.CommandStore;
import protoj.core.PropertyInfo;
import protoj.core.PropertyStore;

/**
 * Provides support for reporting project information such as help and
 * versioning.
 * 
 * @author Ashley Williams
 * 
 */
public final class InfoFeature {
	/**
	 * See {@link #InfoFeature(StandardProject, String)}.
	 */
	private final StandardProject project;

	/**
	 * See {@link #InfoFeature(StandardProject, String)}.
	 */
	private final String versionInfo;

	/**
	 * Accepts the owning project and the information to be reported by
	 * {@link #reportVersion()},
	 * 
	 * @param project
	 * @param versionInfo
	 */
	public InfoFeature(StandardProject project, String versionInfo) {
		this.project = project;
		this.versionInfo = versionInfo;
	}

	/**
	 * Reports general project help information to stdout by iterating over the
	 * commands in the {@link CommandStore} and properties in the
	 * {@link PropertyStore} and using their descriptive properties.
	 * 
	 */
	public void reportProjectHelp() {
		String scriptName = project.getLayout().getShellScript().getName();
		StringBuilder builder = new StringBuilder();
		builder.append("Usage: ");
		builder.append(scriptName);
		builder.append(" [commands]");
		builder
				.append("\n\n* Use the command 'help <subcommand>' to see the help on a specific command.");
		builder
				.append("\n\n* Use the command 'version' to see version information.");
		builder
				.append("\n\n* Use the command 'opts' to specify system properties, for example: \"opts -Dmyprop=myvalue\"");
		builder
				.append("\n\n* If multiple commands are specified, they must be separated with space");
		builder.append("\ncharacters and are executed in order.");
		builder
				.append("\n\n* Where a command accepts arguments, enclose the command and its arguments");
		builder.append("\nin quotes, for example: \"tar -nosrc\".");
		builder.append("\n\nAvailable commands:");
		Collection<Command> commands = project.getCommandStore().getCommands();
		for (Command command : commands) {
			builder.append("\n   ");
			builder.append(command.getAliasText());
		}
		builder.append("\n\nAvailable system properties:");
		Collection<PropertyInfo> properties = project.getPropertyStore()
				.getInfos();
		for (PropertyInfo property : properties) {
			builder.append("\n   ");
			builder.append(property.getKey());
		}
		builder.append("\n\nHere is an example of a call:");
		builder.append("\n\n   ");
		builder.append(scriptName);
		builder
				.append(" compile assemble \"tar -nosrc\" \"opts -Dprotoj.quiet\"");
		builder.append("\n");
		System.out.println(builder);
	}

	/**
	 * Reports help on the specified command to stdout.
	 * 
	 * @param command
	 */
	public void reportCommandHelp(Command command) {
		System.out.println(command.getHelpText());
	}

	/**
	 * Reports help on the specified property to stdout.
	 * 
	 * @param property
	 */
	public void reportPropertyHelp(PropertyInfo property) {
		StringBuilder builder = new StringBuilder();
		builder.append("Name: ");
		builder.append(property.getKey());
		builder.append("\n\nDescription: ");
		builder.append(property.getDescription());
		builder
				.append("\n\nWhen the key is not specified the application assumes a value of '");
		builder.append(property.getMissingDefault());
		builder.append("'");
		builder
				.append("\nWhen the key is specified, but not assigned a value, the application assumes a value of '");
		builder.append(property.getEmptyDefault());
		builder.append("'");
		builder.append("\n");
		System.out.println(builder);
	}

	/**
	 * Reports version information to stdout.
	 */
	public void reportVersion() {
		System.out.println(versionInfo);
	}
}
