/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.lang;

import protoj.core.CommandStore;
import protoj.core.internal.AjcCommand;
import protoj.core.internal.CleanCommand;
import protoj.core.internal.CompileCommand;
import protoj.core.internal.JavacCommand;
import protoj.core.internal.RetrieveCommand;
import protoj.lang.internal.ArchiveCommand;
import protoj.lang.internal.ConfigureCommand;
import protoj.lang.internal.HelpCommand;
import protoj.lang.internal.JunitCommand;
import protoj.lang.internal.PublishCommand;
import protoj.lang.internal.ScpCommand;
import protoj.lang.internal.TarCommand;
import protoj.lang.internal.VerifyTarCommand;
import protoj.lang.internal.VersionCommand;

/**
 * These are all the commands that are added to the {@link CommandStore} on
 * behalf of the {@link StandardProject}.
 * 
 * @author Ashley Williams
 * 
 */
public final class StandardCommands {

	/**
	 * See {@link #getPublishCommand()}.
	 */
	private PublishCommand publishCommand;

	/**
	 * See {@link #getHelpCommand()}.
	 */
	private HelpCommand helpCommand;

	/**
	 * See {@link #getVersionCommand()}.
	 */
	private VersionCommand versionCommand;

	/**
	 * See {@link #getJunitCommand()}.
	 */
	private JunitCommand junitCommand;

	/**
	 * See {@link #getArchiveCommand()}.
	 */
	private ArchiveCommand archiveCommand;

	/**
	 * See {@link #getTarCommand()}.
	 */
	private TarCommand tarCommand;

	/**
	 * See {@link #getVerifyTarCommand()}.
	 */
	private VerifyTarCommand verifyTarCommand;

	/**
	 * See {@link #getScpCommand()}.
	 */
	private ScpCommand scpCommand;

	/**
	 * See {@link #getConfigureCommand()}.
	 */
	private ConfigureCommand configureCommand;

	/**
	 * See {@link #StandardCommands(StandardProject)}.
	 */
	private final StandardProject parent;

	/**
	 * Create an instance with the owning parent project.
	 * 
	 * @param parent
	 */
	public StandardCommands(StandardProject parent) {
		this.parent = parent;
		this.helpCommand = new HelpCommand(parent);
		this.publishCommand = new PublishCommand(parent);
		this.versionCommand = new VersionCommand(parent);
		this.junitCommand = new JunitCommand(parent);
		this.archiveCommand = new ArchiveCommand(parent);
		this.tarCommand = new TarCommand(parent);
		this.verifyTarCommand = new VerifyTarCommand(parent);
		this.scpCommand = new ScpCommand(parent);
		this.configureCommand = new ConfigureCommand(parent);
	}

	/**
	 * Use this command to copy project artifacts with the scp protocol.
	 * 
	 * @return
	 */
	public ScpCommand getScpCommand() {
		return scpCommand;
	}

	/**
	 * Use this command to apply profiles to the project.
	 * 
	 * @return
	 */
	public ConfigureCommand getConfigureCommand() {
		return configureCommand;
	}

	/**
	 * Use this command to report parent information such as version strings and
	 * help text to stdout.
	 * 
	 * @return
	 */
	public HelpCommand getHelpCommand() {
		return helpCommand;
	}

	/**
	 * Use this command to report version information to stdout.
	 * 
	 * @return
	 */
	public VersionCommand getVersionCommand() {
		return versionCommand;
	}

	/**
	 * Use this command to clean the generated content, not including the log
	 * file.
	 * 
	 * @return
	 */
	public CleanCommand getCleanCommand() {
		return parent.getCore().getCleanCommand();
	}

	/**
	 * Use this command to create one or more jar files from the parent
	 * resources and compiled classes.
	 * 
	 * @return
	 */
	public ArchiveCommand getArchiveCommand() {
		return archiveCommand;
	}

	/**
	 * Use this command to exercise all junit tests under the source directory.
	 * 
	 * @return
	 */
	public JunitCommand getJunitCommand() {
		return junitCommand;
	}

	/**
	 * Use this command to make a compressed tar of the parent.
	 * 
	 * @return
	 */
	public TarCommand getTarCommand() {
		return tarCommand;
	}

	/**
	 * Use this command to verify the correctness of a created parent tar.
	 * 
	 * @return
	 */
	public VerifyTarCommand getVerifyTarCommand() {
		return verifyTarCommand;
	}

	/**
	 * Use this command to access the functionality for downloading
	 * dependencies.
	 * 
	 * @return
	 */
	public RetrieveCommand getRetrieveCommand() {
		return parent.getCore().getRetrieveCommand();
	}

	/**
	 * Use this command to perform in-code compilation.
	 * 
	 * @return
	 */
	public CompileCommand getCompileCommand() {
		return parent.getCore().getCompileCommand();
	}

	/**
	 * Use this command to invoke javac usually from a script to compile just
	 * the bootstrap portion of a project.
	 * 
	 * @return
	 */
	public JavacCommand getJavacCommand() {
		return parent.getCore().getJavacCommand();
	}

	/**
	 * Use this command to invoke ajc usually from a script to compile just the
	 * bootstrap portion of a project.
	 * 
	 * @return
	 */
	public AjcCommand getAjcCommand() {
		return parent.getCore().getAjcCommand();
	}

	/**
	 * Use this command to publish parent artifacts.
	 * 
	 * @return
	 */
	public PublishCommand getPublishCommand() {
		return publishCommand;
	}

}
