package protoj.lang.internal;

import java.io.File;
import java.util.Arrays;

import joptsimple.ArgumentAcceptingOptionSpec;
import joptsimple.OptionParser;
import joptsimple.OptionSet;
import joptsimple.OptionSpecBuilder;
import protoj.lang.SampleProjectFeature;
import protoj.lang.StandardProject;

/**
 * This is the entry point for the protoj executable jar.
 * 
 * @author Ashley Williams
 * 
 */
public final class ProtoExecutableMain {
	/**
	 * Performs the following jobs depending on the args parameter:
	 * <ul>
	 * <li>--sample option specified - causes a sample project with the option
	 * name to be created in the current directory.</li>
	 * <li>--help - prints help to stdout</li>
	 * <li>[no options] - arguments are assumed to be commands that are
	 * dispatched to a new project</li>
	 * </ul>
	 * 
	 * @param args
	 *            not used
	 */
	public static void main(String[] args) {
		new ProtoExecutableMain(args);
	}

	private ArgumentAcceptingOptionSpec<String> sampleOption;
	private OptionSpecBuilder helpOption;
	private OptionParser parser;
	private OptionSet options;
	private String[] args;

	public ProtoExecutableMain(String[] args) {
		this.args = args;
		this.parser = new OptionParser();

		String createDescription = String
				.format(
						"specify the name of a sample project to create, can be one of: %s or %s",
						"basic", "ajc");
		this.sampleOption = parser.accepts("sample", createDescription)
				.withRequiredArg().ofType(String.class);
		sampleOption.describedAs("project name");

		this.helpOption = parser.acceptsAll(Arrays.asList("help", "?"),
				"show help");

		this.options = parser.parse(args);

		if (options.has(helpOption)) {
			parser.printHelpOn(System.out);
		} else if (options.has(sampleOption)) {
			createSampleProject(options.valueOf(sampleOption));
		} else {
			dispatchCommands();
		}
	}

	private void dispatchCommands() {
		StandardProject standardProject = new StandardProject(args, null);
		standardProject.getDispatchFeature().dispatchCommands();
	}

	private void createSampleProject(String projectName) {
		File jarFile = new File(ProtoExecutableMain.class.getProtectionDomain()
				.getCodeSource().getLocation().toURI());
		StandardProject project = new StandardProject();
		SampleProjectFeature feature = project.getSampleProjectFeature();
		if (projectName.equals("basic")) {
			feature.createBasicProject(new File("."), jarFile);
		} else if (projectName.equals("ajc")) {
			feature.createAjcProject(new File("."), jarFile);
		} else {
			System.out.println("unrecognized project name: " + projectName);
			parser.printHelpOn(System.out);
		}
	}
}
