/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.lang.internal;

import java.io.File;

import joptsimple.OptionParser;
import joptsimple.OptionSet;
import joptsimple.OptionSpec;
import protoj.core.Command;
import protoj.core.CommandStore;
import protoj.core.ProjectLayout;
import protoj.lang.ScpFeature;
import protoj.lang.StandardProject;

/**
 * Command driven access to {@link ScpFeature}.
 * 
 * @author Ashley Williams
 * 
 */
public final class ScpCommand {

	private final class Body implements Runnable {

		public void run() {
			Command delegate = getDelegate();
			OptionSet options = delegate.getOptions();
			String noKey = options.valueOf(getNoKey());
			String key = options.valueOf(getKey());
			File keyFile = options.valueOf(getKeyFile());
			String passwordText = options.valueOf(getPasswordText());
			if (noKey != null) {
				project.getScpFeature().copyProjectArchive(noKey, passwordText);
			} else if (key != null) {
				project.getScpFeature().copyProjectArchive(key, passwordText,
						keyFile.getAbsolutePath());
			} else {
				StringBuilder builder = new StringBuilder();
				builder.append("missing mandatory option ");
				builder.append(getNoKey().toString());
				builder.append(" or ");
				builder.append(getKey().toString());
				delegate.throwBadOptionsException(builder.toString());
			}
		}
	}

	private OptionSpec<String> noKey;
	private OptionSpec<String> key;
	private OptionSpec<String> passwordText;
	private OptionSpec<File> keyFile;
	private Command delegate;
	private final StandardProject project;

	public ScpCommand(StandardProject project) {
		this.project = project;
		ProjectLayout layout = project.getLayout();
		CommandStore store = project.getCommandStore();

		StringBuilder description = new StringBuilder();
		description.append("Copies the project tar file to a remote location.");
		description
				.append("\nIf no tar file has been generated then this delegate will fail.");
		description.append("\nThe following arguments are required:");
		description
				.append("\n\n   1. -nokey <username>@<host>:<dest> -w <password> - where simple password");
		description.append("\n      authentication is required");
		description
				.append("\n\n   2. -key <username>@<host>:<dest> -i <keyfile> -w <passphrase> - where key-based");
		description.append("\n      authentication is required.");
		description.append("\n\nExample: jonny$ ./");
		description.append(layout.getScriptName());
		description
				.append(" \"scp -nokey jonny123@acme.org:~/jonny123 -w abcd1234\"");
		description.append("\n\nExample: jonny$ ./");
		description.append(layout.getScriptName());
		description.append(" \"scp -nokey jonny123@acme.org:~/jonny123\"");
		description.append("\n\nExample: jonny$ ./");
		description.append(layout.getScriptName());
		description
				.append(" \"scp -key jonny123@acme.org:~/jonny123 -i /Users/jonny/.ssh/id_dsa -w abcd1234\"");
		description.append("\n\nExample: jonny$ ./");
		description.append(layout.getScriptName());
		description
				.append(" \"scp -key jonny123@acme.org:~/jonny123 -w abcd1234\"");
		delegate = store.addCommand("scp", description.toString(), "16m",
				new Body());
		OptionParser parser = delegate.getParser();
		noKey = parser.accepts("nokey").withRequiredArg().ofType(String.class);
		key = parser.accepts("key").withRequiredArg().ofType(String.class);
		passwordText = parser.accepts("w").withRequiredArg().ofType(
				String.class);
		keyFile = parser.accepts("i").withRequiredArg().ofType(File.class);
	}

	public OptionSpec<String> getNoKey() {
		return noKey;
	}

	public OptionSpec<String> getKey() {
		return key;
	}

	public OptionSpec<String> getPasswordText() {
		return passwordText;
	}

	public OptionSpec<File> getKeyFile() {
		return keyFile;
	}

	public Command getDelegate() {
		return delegate;
	}
}
