/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.lang.internal.acme;

import java.io.File;

import protoj.core.ProjectLayout;
import protoj.lang.ScriptSession;
import protoj.lang.StandardProject;
import protoj.lang.internal.ProtoProject;

/**
 * In a nutshell this represents the high level test that puts the acme project
 * through its paces. It uses a {@link ScriptSession} to run a variety of
 * commands and supplies matching listeners in order to verify correct
 * functionality with assertions.
 * 
 * @author Ashley Williams
 * 
 */
public final class AcmeSession {
	/**
	 * See {@link #getProject()}.
	 */
	private StandardProject project;

	/**
	 * After this constructor call the physical acme project will have been
	 * created.
	 */
	public AcmeSession() {
		ProtoProject protoProject = new ProtoProject();
		protoProject.createNoDepArchive();
		this.project = protoProject.createAcmeProjectDelegate();
	}

	/**
	 * Kicks off the command executions.
	 */
	public void execute() {
		String acmeFile = getAcmeFile().getAbsolutePath();
		ScriptSession session = project.createScriptSession();

		session.addCommand("help", new AssertProjectHelp(this),
				AssertProjectHelp.WARN);
		session.addCommand("", new AssertCompile(this), AssertCompile.EMPTY);
		session.addCommand("\"compile -i acme/util\"", new AssertCompile(this),
				AssertCompile.UTIL);
		session.addCommand("compile", new AssertCompile(this),
				AssertCompile.ALL);
		session.addCommand("fail", new AssertFail(this), null);
		session.addCommand("retrieve", new AssertCompile(this),
				AssertCompile.RETRIEVE);
		session.addCommand("\"config -name test\"", new AssertConfigure(this),
				AssertConfigure.CONFIG);
		session.addCommand(
				"\"scp --nokey http://someserver:destdir -w somepassword\"",
				new AssertScp(this), AssertScp.NO_TAR);
		session.addCommand("junit", new AssertJunit(this), null);
		session.addCommand("test", new AssertJunit(this), null);
		session.addCommand("help", new AssertProjectHelp(this),
				AssertProjectHelp.OK);
		session.addCommand("\"help compile\"", new AssertCommandHelp(this),
				null);
		session.addCommand("version", new AssertVersion(this), null);
		session.addCommand("archive", new AssertArchive(this), null);
		session.addCommand("\"publish -artifact foo\"",
				new AssertPublish(this), AssertPublish.FOO);
		session.addCommand("publish", new AssertPublish(this),
				AssertPublish.ALL);
		session.addCommand("verify-tar", new AssertVerifyTar(this),
				AssertVerifyTar.NO_TAR);
		session.addCommand("\"tar -nosrc\"", new AssertTar(this), null);
		session.addCommand("verify-tar", new AssertVerifyTar(this),
				AssertVerifyTar.NO_SRC);
		session.addCommand("tar", new AssertTar(this), null);
		session.addCommand("verify-tar", new AssertVerifyTar(this),
				AssertVerifyTar.OK);
		session.addCommand("\"scp\"", new AssertScp(this), AssertScp.NO_ARGS);
		session.addCommand(
				"\"scp --nokey http://someserver:destdir -w somepassword\"",
				new AssertScp(this), AssertScp.OK);
		session.addCommand(String.format("create-file \"opts -Dacme.file=%s\"",
				acmeFile), new AssertJvmarg(this), null);
		session.addCommand(
				"help \"opts -Dprotoj.debug -Dprotoj.debug.port=12345\"",
				new AssertDebug(this), AssertDebug.PORT_12345);
		session.addCommand(
				"help \"opts -Dprotoj.debug -Dprotoj.debug.condition\"",
				new AssertDebug(this), AssertDebug.PORT_DEFAULT);
		session.addCommand(
				"help \"opts -Dprotoj.debug -Dprotoj.debug.condition=help\"",
				new AssertDebug(this), AssertDebug.VERSION_CMD);
		session
				.addCommand(
						"help \"opts -Dprotoj.debug -Dprotoj.debug.condition=garbage\"",
						new AssertDebug(this), AssertDebug.NO_CMD);
		session
				.addCommand(
						"junit \"opts -Dprotoj.debug -Dprotoj.debug.condition=junit-task\"",
						new AssertDebug(this), AssertDebug.JUNIT_TASK);
		session
				.addCommand(
						"message \"opts -Dprotoj.debug -Dprotoj.debug.condition=project\"",
						new AssertDebug(this), AssertDebug.PROJECT);
		session.addCommand("v \"opts -Dprotoj.debug.port=11111\"",
				new AssertDebug(this), AssertDebug.NO_DEBUG);
		session.addCommand("\"config -name test -undo\"", new AssertConfigure(
				this), AssertConfigure.UNDO);
		session.addCommand("clean", new AssertClean(this), null);
		session.execute();
	}

	/**
	 * Similar to the real acme project.
	 * 
	 * @return
	 */
	public StandardProject getProject() {
		return project;
	}

	/**
	 * Represents a file that should exist after the corresponding command has
	 * been executed.
	 * 
	 * @return
	 */
	public File getAcmeFile() {
		ProjectLayout acmeLayout = getProject().getLayout();
		return new File(acmeLayout.getTargetDir(), "testvmarg.txt");
	}
}
