/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.lang.internal.acme;

import org.junit.Assert;

import protoj.core.ArgRunnable;
import protoj.core.ProjectLayout;
import protoj.lang.ScriptSession;

/**
 * These assertions trigger when the various debug properties are specified and
 * ensures that they perform as described in the project help.
 * 
 * @author Ashley Williams
 * 
 */
final class AssertDebug implements ArgRunnable<ScriptSession> {
	public static final String PORT_12345 = "port 12345";

	public static final String PORT_DEFAULT = "port default";

	public static final String VERSION_CMD = "version cmd";

	public static final String NO_CMD = "no cmd";

	public static final String JUNIT_TASK = "junit task";

	public static final String NO_DEBUG = "no debug";

	public static final String PROJECT = "project";

	private final AcmeSession acmeSession;

	/**
	 * This listener is attached to more than one debug command so we can keep
	 * track of which one with this testIndex. Changing the order of the
	 * commands in {@link AcmeSession} would break these assertion tests.
	 */
	private int testIndex = 0;

	public AssertDebug(AcmeSession acmeSession) {
		this.acmeSession = acmeSession;
	}

	public void run(ScriptSession projectSession) {
		ProjectLayout acmeLayout = acmeSession.getProject().getLayout();
		String tag = projectSession.getCurrentTag();
		String stdout = projectSession.getCurrentExec().getStdout();

		String debugText = "Listening for transport dt_socket at address: ";
		String port12345Text = debugText + "12345";
		String port11111Text = debugText + "11111";
		String allCommandsMessage = "for all commands";
		String junitMessage = "for junit tasks";
		String projectMessage = "for project vm";
		String commandMessage = "for command: help";

		if (tag.equals(PORT_12345)) {
			// debug session detected at the 12345 port when specified
			Assert.assertTrue(stdout, stdout.contains(port12345Text));
			Assert.assertTrue(stdout, stdout.contains(allCommandsMessage));
		} else if (tag.equals(PORT_DEFAULT)) {
			// debug session detected at the default port when not specified
			Assert.assertTrue(stdout, stdout.contains(port11111Text));
			Assert.assertTrue(stdout, stdout.contains(allCommandsMessage));
		} else if (tag.equals(VERSION_CMD)) {
			// debug session detected when targeting the 'version command' vm
			Assert.assertTrue(stdout, stdout.contains(port11111Text));
			Assert.assertTrue(stdout, stdout.contains(commandMessage));
		} else if (tag.equals(NO_CMD)) {
			// no debug session when not targeting the 'version command' vm
			Assert.assertFalse(stdout, stdout.contains(debugText));
			Assert
					.assertFalse(stdout, stdout
							.contains("debug mode configured"));
		} else if (tag.equals(JUNIT_TASK)) {
			// a debug session is detected when targeting the junit test vm
			Assert.assertTrue(stdout, stdout.contains(port11111Text));
			Assert.assertTrue(stdout, stdout.contains(junitMessage));
		} else if (tag.equals(PROJECT)) {
			// a debug session is detected when targeting the project vm
			// which is achieved by executing a command not known in the
			// core vm - basically any newly added commands in the project vm
			Assert.assertTrue(stdout, stdout.contains(port11111Text));
			Assert.assertTrue(stdout, stdout.contains(projectMessage));
		} else if (tag.equals(NO_DEBUG)) {
			// no debug session when not no protoj.debug property present
			Assert.assertFalse(stdout, stdout.contains(debugText));
		}

		acmeLayout.getLogFile().delete();
		testIndex++;
	}
}