/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.lang.internal.ant;

import java.io.File;

import org.apache.tools.ant.taskdefs.ExecTask;
import org.apache.tools.ant.types.Commandline.Argument;

import protoj.core.internal.AntTarget;

/**
 * A convenience class for executing a system command. Use the constructors to
 * specify the minimal and most widely anticipated configuration and the
 * <code>initXXX</code> methods for the less common configuration options.
 * <p>
 * As far as possible this class is used to hide the differences in syntax
 * between different operating systems.
 * 
 * @author Ashley Williams
 * 
 */
public final class CommandTask {
	private String resultProperty = "protoj.result";
	private String stdoutProperty = "protoj.stdout";
	private String stderrProperty = "protoj.stderr";
	private AntTarget target;
	private ExecTask execTask;

	/**
	 * Accepts the details of the system command that should be executed. This
	 * constructor also takes care of ensuring that on windows, the executable
	 * is in fact <code>cmd.exe</code>.
	 * 
	 * @param dir
	 *            the directory in which the command should be executed
	 * @param executable
	 *            the command to execute without any command line arguments
	 * @param argLine
	 *            a space-delimited list of command-line arguments.
	 */
	public CommandTask(File dir, String executable, String argLine) {
		this.target = new AntTarget("protoj-execTask");

		execTask = new ExecTask();
		target.addTask(execTask);
		execTask.setDir(dir);

		Argument arg = execTask.createArg();

		if (isWindows()) {
			arg.setLine("/c " + executable + " " + argLine);
			execTask.setExecutable("cmd.exe");
		} else {
			arg.setLine(argLine);
			execTask.setExecutable(executable);
		}
	}

	public void initSpawn(boolean spawn) {
		execTask.setSpawn(spawn);
		if (!spawn) {
			execTask.setOutputproperty(stdoutProperty);
			execTask.setErrorProperty(stderrProperty);
			execTask.setResultProperty(resultProperty);
		}
	}

	/**
	 * Helper for {@link #CommandTask} for determining if the current OS is
	 * windows.
	 * 
	 * @return
	 */
	public boolean isWindows() {
		String osName = System.getProperty("os.name").toUpperCase();
		return osName.indexOf("WINDOWS") >= 0;
	}

	public void execute() {
		target.execute();
	}

	public String getStdout() {
		return target.getProject().getProperty(stdoutProperty);
	}

	public String getStderr() {
		return target.getProject().getProperty(stderrProperty);
	}

	public String getResult() {
		return target.getProject().getProperty(resultProperty);
	}

	/**
	 * All the output from the executed command seems to get buffered.command
	 * has exited, the buffered output can be written using this method.
	 */
	public void writeOutput() {
		String systemOut = getStdout();
		if (systemOut != null && systemOut.length() > 0) {
			System.out.println(systemOut);
		}
		String systemErr = getStderr();
		if (systemErr != null && systemErr.length() > 0) {
			System.err.println(systemErr);
		}
	}

	/**
	 * It only makes sense to call this method from a forked but non-spawned vm.
	 * Remember that a spawned vm will outlive the calling process.
	 * 
	 * @return
	 */
	public boolean isSuccess() {
		return getResult().equals("0");
	}
}
