/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.lang.internal.basic;

import java.io.File;

import org.junit.Assert;

import protoj.core.ArgRunnable;
import protoj.core.ProjectLayout;
import protoj.lang.ScriptSession;
import protoj.lang.StandardProject;
import protoj.lang.internal.ProtoProject;
import protoj.lang.internal.ant.CommandTask;

/**
 * Executes a selection of commands against the basic project and performs
 * various asserts against their results.
 * 
 * @author Ashley Williams
 * 
 */
public final class BasicSession {
	private static final String COMPILE = "compile";
	private static final String RETRIEVE = "retrieve";
	private static final String HELLO_BASIC = "hello-basic";
	private static final String HELP = "help";

	private StandardProject project;

	/**
	 * After this constructor call the physical basic project will have been
	 * created.
	 */
	public BasicSession() {
		ProtoProject protoProject = new ProtoProject();
		this.project = protoProject.createSampleProjectDelegate("basic", false);
	}

	public void execute() {
		ScriptSession session = project.createScriptSession();
		ArgRunnableImplementation listener = new ArgRunnableImplementation();
		session.addCommand("compile", listener, COMPILE);
		session.addCommand("retrieve", listener, RETRIEVE);
		session.addCommand("help", listener, HELP);
		session.addCommand("hello-basic", listener, HELLO_BASIC);
		session.execute();
	}

	private final class ArgRunnableImplementation implements
			ArgRunnable<ScriptSession> {
		public void run(ScriptSession session) {
			ProjectLayout layout = project.getLayout();
			CommandTask exec = session.getCurrentExec();
			String stdout = exec.getStdout();
			project.getLogger().debug(stdout);
			String log = project.getLayout().loadLog();

			String tag = session.getCurrentTag();
			if (tag.equals(RETRIEVE)) {
				File libDir = layout.getLibDir();
				File javadoc = new File(libDir, "junit-4.4-javadoc.jar");
				File sources = new File(libDir, "junit-4.4-sources.jar");
				File classes = new File(libDir, "junit-4.4.jar");
				Assert.assertTrue(javadoc.exists());
				Assert.assertTrue(sources.exists());
				Assert.assertTrue(classes.exists());
				Assert.assertTrue(log, exec.isSuccess());
			} else if (tag.equals(COMPILE)) {
				File basicProjectClass = new File(layout.getClassesDir(),
						"org/basic/system/BasicProject.class");
				Assert.assertTrue(basicProjectClass.exists());
				Assert.assertTrue(log, exec.isSuccess());
			} else if (tag.equals(HELP)) {
				Assert.assertTrue(stdout, stdout.contains("hello-basic"));
				Assert.assertTrue(log, exec.isSuccess());
			} else if (tag.equals(HELLO_BASIC)) {
				Assert.assertTrue(stdout, stdout.contains("hello from this sample project!"));
				Assert.assertTrue(log, exec.isSuccess());
			}

			layout.getLogFile().delete();
		}
	}

}
