/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.core;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;

import protoj.core.internal.CoreProject;

/**
 * A repository of the application commands. A command can only be created by
 * calling the {@link #addCommand(String, String, String, Runnable)}, which
 * automatically ensures that command is added to this store.
 * 
 * @author Ashley Williams
 * 
 */
public final class CommandStore {
	/**
	 * See {@link #getCommands()}.
	 */
	private ArrayList<Command> commands = new ArrayList<Command>();

	/**
	 * See {@link #CommandStore(CoreProject)}.
	 */
	private final CoreProject core;

	/**
	 * Accepts the parent of this instance.
	 * 
	 * @param core
	 */
	public CommandStore(final CoreProject core) {
		this.core = core;
	}

	/**
	 * The parent core for this store.
	 * 
	 * @return
	 */
	public CoreProject getCore() {
		return core;
	}

	/**
	 * Adds a new command with the specified command body and that can execute
	 * in a new vm. See
	 * {@link Command#Command(CommandStore, String, String, String, Runnable)}
	 * for a description of the parameters.
	 * 
	 * @param name
	 * @param description
	 * @param maxMemory
	 * @param body
	 * @return
	 */
	public Command addCommand(String name, String description,
			String maxMemory, Runnable body) {
		Command command = new Command(this, name, description, maxMemory, body);
		commands.add(command);
		return command;
	}

	/**
	 * Read-only access to the available commands.
	 * 
	 * @return
	 */
	public Collection<Command> getCommands() {
		return Collections.unmodifiableCollection(commands);
	}

	/**
	 * Looks up the command with the given name. If no match is found then null
	 * is returned.
	 * 
	 * @param name
	 * @return
	 */
	public Command getCommand(String name) {
		Command targetCommand = null;
		Collection<Command> values = commands;
		for (Command command : values) {
			if (command.containsAlias(name)) {
				targetCommand = command;
				break;
			}
		}
		return targetCommand;
	}

}
