/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.core.internal;

import java.util.List;

import joptsimple.OptionSet;
import joptsimple.OptionSpec;
import protoj.core.Command;
import protoj.core.CommandStore;

/**
 * Compiles the project code using the ajc compiler. Typically this command is
 * invoked from inside a shell script as an alternative to invoking the ajc
 * compiler in an os-specific way. For example the logic that detects whether or
 * not the source directory exists would be hard to replicate across batch files
 * and bash scripts.
 * <p>
 * In order to configure the underlying ant task, beans introspection is used on
 * the protoj.core.AspectjCompileTask parent. First an example of an ajc command
 * that configures the ant source property and includes filter using the protoj
 * -opt option that accepts name=value pairs:
 * 
 * <pre>
 * &quot;ajc -opt compileTask.source=1.5 -opt dirSet.includes=org/foo/core&quot;
 * </pre>
 * 
 * That parent contains two properties called compileTask and dirSet that are
 * instances of org.aspectj.tools.ant.taskdefs.AjcTask and
 * org.apache.tools.ant.types.DirSet respectively and so consult the ant and
 * aspectj reference to determine the available set of properties that are
 * available for configuration.
 * 
 * @author Ashley Williams
 * 
 */
public final class AjcCommand {
	private final class Body implements Runnable {
		public void run() {
			OptionSet options = delegate.getOptions();
			if (options.has(optOption)) {
				List<String> compilerOptions = options.valuesOf(optOption);
				parent.getCompileFeature().compileAjc(compilerOptions);
			}
		}
	}

	/**
	 * Provides the basic command functionality.
	 */
	private Command delegate;

	/**
	 * The parent of this command.
	 */
	private final CoreProject parent;

	private OptionSpec<String> optOption;

	public AjcCommand(CoreProject core) {
		this.parent = core;
		CommandStore store = core.getCommandStore();

		StringBuilder description = new StringBuilder();

		description
				.append("\nCompiles the project code using the ajc compiler. Typically this command is");
		description
				.append("\ninvoked from inside a shell script as an alternative to invoking the ajc");
		description
				.append("\ncompiler in an os-specific way. For example the logic that detects whether or");
		description
				.append("\nnot the source directory exists would be hard to replicate across batch files");
		description.append("\nand bash scripts.");
		description.append("\n");
		description
				.append("\nIn order to configure the underlying ant task, beans introspection is used on");
		description
				.append("\nthe protoj.core.AspectjCompileTask parent. First an example of an ajc command");
		description
				.append("\nthat configures the ant source property and includes filter using the protoj");
		description.append("\n-opt option that accepts name=value pairs:");
		description.append("\n");
		description
				.append("\n\"ajc -opt compileTask.source=1.5 -opt dirSet.includes=org/foo/core\"");
		description.append("\n");
		description
				.append("\nThat parent contains two properties called compileTask and dirSet that are");
		description
				.append("\ninstances of org.aspectj.tools.ant.taskdefs.AjcTask and");
		description
				.append("\norg.apache.tools.ant.types.DirSet respectively and so consult the ant");
		description
				.append("\nand aspectj reference to determine the available set of properties that");
		description.append("\nare available for configuration.");
		delegate = store.addCommand("ajc", description.toString(), "16m",
				new Body());
		delegate.initBootstrapCurrentVm();
		optOption = delegate.getParser().accepts("opt").withRequiredArg();
	}

	public Command getDelegate() {
		return delegate;
	}

}
