/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.core.internal;

import org.apache.log4j.Level;

import protoj.core.ArgRunnable;
import protoj.core.Command;
import protoj.core.DispatchFeature;
import protoj.core.PropertyInfo;
import protoj.core.ProtoLogger;
import protoj.core.StandardProperties;

/**
 * Implements core configuration based on system properties for the whole of
 * protoj.
 * 
 * @author Ashley Williams
 * 
 */
public aspect CoreConfig {
	/**
	 * Shows or hides the status window when the dispatch feature has finished
	 * dispatching all the commands.
	 * 
	 * @param project
	 */
	after(CoreProject project) returning : execution(CoreProject.new(..)) && this(project) {
		PropertyInfo statusWindow = project.getProperties()
				.getStatusWindowProperty();
		project.getDispatchFeature().setStatusWindow(
				statusWindow.getBooleanValue());
	}

	/**
	 * Every time the dispatch feature bootstraps to the project vm, we
	 * intervene in order to specify debug options.
	 * 
	 * @param command
	 * @param project
	 */
	before(final DispatchFeature feature) : execution(* startProjectMainVm(..)) && this(feature) {
		feature.setStartVmConfig(new ArgRunnable<StartVmCommand>() {
			public void run(StartVmCommand vm) {
				CoreProject project = feature.getProject();
				StandardProperties properties = project.getProperties();

				// only check other properties if debugging is switched on
				if (properties.getDebugProperty().getBooleanValue()) {
					String condition = properties.getDebugConditionProperty()
							.getValue();
					int port = properties.getDebugPortProperty()
							.getIntegerValue();
					boolean suspend = properties.getDebugSuspendProperty()
							.getBooleanValue();

					if (condition != null && condition.equals("project")) {
						vm.initDebug(port, suspend);
						String message = "debug mode configured for project vm at port "
								+ port;
						project.getLogger().info(message);
					}
				}
			}
		});
	}

	/**
	 * Every time a command is executed we intervene in order to specify debug
	 * options.
	 * 
	 * @param command
	 */
	before(final Command command) : execution(* startVm(..)) && this(command) {
		command.setStartVmConfig(new ArgRunnable<StartVmCommand>() {
			public void run(StartVmCommand vm) {
				CoreProject project = command.getStore().getCore();
				StandardProperties properties = project.getProperties();
				// only check other properties if debugging is switched on
				if (properties.getDebugProperty().getBooleanValue()) {
					String condition = properties.getDebugConditionProperty()
							.getValue();
					int port = properties.getDebugPortProperty()
							.getIntegerValue();
					boolean suspend = properties.getDebugSuspendProperty()
							.getBooleanValue();
					if (command.containsAlias(condition)) {
						vm.initDebug(port, suspend);
						String message = "debug mode configured at port "
								+ port + " for command: " + condition;
						project.getLogger().info(message);
					} else if (condition.equals("")) {
						vm.initDebug(port, suspend);
						String message = "debug mode configured at port "
								+ port + " for all commands";
						project.getLogger().info(message);
					}
				}
			}
		});
	}

	/**
	 * When the project is created, the logging level and console appender
	 * configuration is modified depending on the system properties in the
	 * property store.
	 * 
	 * @param project
	 */
	after(CoreProject project) returning : execution(CoreProject.new(..)) && this(project) {
		StandardProperties properties = project.getProperties();
		ProtoLogger logger = project.getProtoLogger();
		boolean quiet = properties.getQuietProperty().getBooleanValue();
		logger.setConsoleAppenderEnabled(!quiet);
		String level = properties.getLevelProperty().getValue();
		logger.setLevel(Level.toLevel(level));
	}

	/**
	 * Configures the DispatchFeature for showing the status window based on the
	 * PropertyStore statusWindowProperty value.
	 * 
	 * @param feature
	 */
	after(CoreProject project) returning : execution(CoreProject.new(..)) && this(project) {
		PropertyInfo statusWindow = project.getProperties()
				.getStatusWindowProperty();
		project.getDispatchFeature().setStatusWindow(
				statusWindow.getBooleanValue());
	}

}
