/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.core.internal;

import java.io.File;

import protoj.core.ArgRunnable;
import protoj.core.CompileFeature;

/**
 * Represents the use-cases available to be carried out against the protoj core.
 * Some use-cases will only be available after the compilation use-case has been
 * invoked. See <code>StandardProject</code> and <code>ProtoProject</code> for
 * further information.
 * 
 * @author Ashley Williams
 * 
 */
public final class ProtoCore {
	/**
	 * See {@link ProtoCore}.
	 */
	public static void main(String[] args) {
		new ProtoCore(args).dispatchCommands();
	}

	/**
	 * Provides standard features such as compilation and assembly.
	 */
	private CoreProject core;

	/**
	 * See the
	 * <code>StandardProject.StandardProject(String[], String, String)</code>
	 * constructor for a discussion of the arguments to main().
	 * 
	 * @param args
	 */
	public ProtoCore(String[] args) {
		this(new CoreProject(args));
	}

	public ProtoCore(File rootDir, String scriptName) {
		this(new CoreProject(rootDir, scriptName));
	}

	public ProtoCore(CoreProject core) {
		this.core = core;
		// choose to support the following features
		core.getCompileFeature().initConfig(true,
				new ArgRunnable<CompileFeature>() {
					public void run(CompileFeature feature) {
						feature.getAjcCompileTask().getCompileTask().setMaxmem(
								"16m");
						feature.getAjcCompileTask().getCompileTask().setSource(
								"1.5");
					}
				});
		core.getDispatchFeature().initBootstrap(
				"protoj.lang.internal.ProtoProject", "16m");
		core.getRetrieveFeature().initConfig("/protoj/ivy.xml", null, null);

		// the commands execute in a new jvm, tune the max memory
		core.getCompileCommand().getDelegate().setMemory("8m");
	}

	/**
	 * See <code>StandardProject#dispatchCommands()</code>.
	 */
	public void dispatchCommands() {
		getCore().getDispatchFeature().dispatchCommands();
	}

	/**
	 * External access to <code>core</code>, just for use by
	 * <code>ProtoProject</code>.
	 * 
	 * @return
	 * @see #core
	 */
	public CoreProject getCore() {
		return core;
	}

	/**
	 * Used in version information and names of jar files for example.
	 * 
	 * @return
	 */
	public String getVersionNumber() {
		return "1.8.4";
	}

}
