/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.lang;

import java.io.File;
import java.io.FilenameFilter;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.TreeMap;

import protoj.core.ArgRunnable;
import protoj.core.ProjectLayout;
import protoj.lang.internal.ant.AssembleTask;

/**
 * Responsible for creating one or more archives from the parent source
 * directory. Call
 * {@link #addArchive(String, String, String, String, ArgRunnable)} to configure
 * each additional archive with the specified name and
 * {@link #createArchive(String)} to create one of the added archive. During
 * creation, the <code>config</code> argument specified in the constructor will
 * be called back to give the caller a chance to provide further archive
 * configuration. Also call {@link #initIncludeArchives(String, String...)} to
 * cause any additional archives from the {@link ProjectLayout#getLibDir()} to
 * be merged with the added archive.
 * 
 * @author Ashley Williams
 * 
 */
public final class SourceArchive {
	/**
	 * The parent that owns this instance lifecycle.
	 */
	private final ArchiveFeature parent;

	/**
	 * The configuration information for each added archive.
	 */
	private TreeMap<String, ArchiveEntry<SourceArchive>> entries = new TreeMap<String, ArchiveEntry<SourceArchive>>();

	/**
	 * See {@link #getCurrentAssembleTask()}.
	 */
	private AssembleTask currentAssembleTask;

	/**
	 * See {@link #getCurrentEntry()}.
	 */
	private ArchiveEntry<SourceArchive> currentEntry;

	/**
	 * Creates with the parent feature.
	 * 
	 * @param parent
	 */
	public SourceArchive(ArchiveFeature parent) {
		this.parent = parent;
	}

	/**
	 * Use to enable creation of an additional archive.
	 * 
	 * @param name
	 *            the name of the jar file to be created, without the extension.
	 * @param manifest
	 *            the name of the manifest to be used from the manifest
	 *            directory, without the extension. Can be null if a default
	 *            manifest is required.
	 * @param includes
	 *            the resources that should be included in the archive, can be
	 *            null to include all resources
	 * @param excludes
	 *            the resources that should be excluded in the archive, can be
	 *            null to exclude no resources
	 * @param config
	 *            this will be called back during the invocatin of
	 *            {@link #createArchive(String)} if the caller wishes to provide
	 *            further configuration. Can be null if the defaults are ok.
	 */

	public void addArchive(String name, String manifest, String includes,
			String excludes, ArgRunnable<SourceArchive> config) {
		String fileName = String.format("%s-%s.jar", name, parent.getLayout()
				.getSourcePostfix());
		ArchiveEntry<SourceArchive> entry = new ArchiveEntry<SourceArchive>(
				name, parent, fileName, manifest, includes, excludes, config);
		entries.put(name, entry);
	}

	/**
	 * Any additional classes archives in the {@link ProjectLayout#getLibDir()}
	 * directory that should get merged during creation of the named archive.
	 * 
	 * @param name
	 * @param archives
	 */
	public void initIncludeArchives(String name, String... archives) {
		getEntry(name).initMergeArchives(archives);
	}

	/**
	 * Specifies that all the sources archives except for the excluded list in
	 * the {@link ProjectLayout#getLibDir()} should get merged during creation
	 * of the named archive.
	 * 
	 * @param name
	 */
	public void initExcludeArchives(String name, String... archives) {
		final List<String> list = Arrays.asList(archives);
		FilenameFilter filter = new FilenameFilter() {
			public boolean accept(File dir, String name) {
				return !list.contains(name) && parent.isClassesJar(name);
			}
		};
		getEntry(name).initMergeArchives(filter);
	}

	/**
	 * Delegates to {@link #createArchive(String)} for each archive that was
	 * added through a call to
	 * {@link #addArchive(String, String, String, String, ArgRunnable)}.
	 */
	public void createArchives() {
		Set<String> keys = entries.keySet();
		for (String key : keys) {
			createArchive(key);
		}
	}

	/**
	 * Creates the jar for the given name under
	 * {@link ProjectLayout#getArchiveDir()}. The jar file will have a name of
	 * [name]-sources.jar.
	 * 
	 * @param name
	 *            the name as specified in the call to
	 *            {@link #addArchive(String, String, String, String, ArgRunnable)}
	 *            . .
	 */
	public void createArchive(String name) {
		ProjectLayout layout = parent.getProject().getLayout();
		currentAssembleTask = entries.get(name).createAssembleTask(
				layout.getJavaDir());
		ArgRunnable<SourceArchive> config = entries.get(name).getConfig();
		if (config != null) {
			config.run(this);
		}
		currentAssembleTask.execute();
	}

	/**
	 * Convenient method that visits each entry in this archive.
	 * 
	 * @param visitor
	 */
	public void visit(ArgRunnable<ArchiveEntry<SourceArchive>> visitor) {
		Collection<ArchiveEntry<SourceArchive>> values = entries.values();
		for (ArchiveEntry<SourceArchive> entry : values) {
			visitor.run(entry);
		}
	}

	/**
	 * Accessor for the entry corresponding to the given name.
	 * 
	 * @param name
	 * @return
	 */
	public ArchiveEntry<SourceArchive> getEntry(String name) {
		return entries.get(name);
	}

	/**
	 * The ant task responsible for creating the jar during the call to
	 * {@link #createArchive(String)}.
	 * 
	 * @return
	 */
	public AssembleTask getCurrentAssembleTask() {
		return currentAssembleTask;
	}

	/**
	 * The instance used to hold information about the the archive being created
	 * during the call to {@link #createArchive(String)}.
	 * 
	 * @return
	 */
	public ArchiveEntry<SourceArchive> getCurrentEntry() {
		return currentEntry;
	}
}