/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.lang.internal;

import joptsimple.ArgumentAcceptingOptionSpec;
import joptsimple.OptionParser;
import joptsimple.OptionSet;
import joptsimple.OptionSpec;
import protoj.core.Command;
import protoj.core.CommandStore;
import protoj.core.ProjectLayout;
import protoj.lang.StandardProject;

/**
 * The command responsible for executing junit tests. The JunitBody instance
 * that implements the command functionality is added later on in the
 * StandardProject constructor.
 * 
 * <pre>
 * Executes the junit tests in the project classes directory. The following
 * options are supported:
 * 
 * 1. -includes: the ant pattern describing the classes that should be executed
 * as junit tests.
 * 
 * 2. -excludes: the ant pattern describing the classes that should not be
 * executed as junit tests.
 * 
 * 3. If neither includes or excludes options are provided then the default
 * behavior applies, which is to execute all tests whose name includes 'Test'.
 * 
 * Example: jonny$ ./protoj.sh test
 * Example: jonny$ ./protoj.sh test -includes ++/MyTest.java
 * (would like to use astrix not +, but it breaks the javadocs)
 * </pre>
 * 
 * @author Ashley Williams
 * 
 */
public final class JunitCommand {
	private final class Body implements Runnable {

		public void run() {
			Command delegate = getDelegate();
			OptionSet options = delegate.getOptions();
			String include = options.valueOf(getIncludes());
			String exclude = options.valueOf(getExcludes());

			if (include == null && exclude == null) {
				project.getJunitFeature().junit();
			} else {
				project.getJunitFeature().junit(include, exclude);
			}
		}
	}

	/**
	 * Provides the basic command functionality.
	 */
	private Command delegate;
	private OptionSpec<String> includes;
	private ArgumentAcceptingOptionSpec<String> excludes;
	private final StandardProject project;

	public JunitCommand(StandardProject project) {
		this.project = project;
		ProjectLayout layout = project.getLayout();
		CommandStore store = project.getCommandStore();
		StringBuilder description = new StringBuilder();
		description
				.append("Executes the junit tests in the project classes directory. The following");
		description.append("\noptions are supported:");
		description
				.append("\n\n   1. -includes: the ant pattern describing the classes that should be executed");
		description.append("\n      as junit tests.");
		description
				.append("\n\n   2. -excludes: the ant pattern describing the classes that should not be");
		description.append("\n      executed as junit tests.");
		description
				.append("\n\n   3. If neither includes or excludes options are provided then the default");
		description
				.append("\n      behavior applies, which is to execute all tests whose name includes 'Test'.");
		description.append("\n\nExample: jonny$ ./");
		description.append(layout.getScriptName());
		description.append(" test");
		description.append("\n\nExample: jonny$ ./");
		description.append(layout.getScriptName());
		description.append(" \"test -includes **/MyTest.java\"");

		delegate = store.addCommand("junit", description.toString(), "16m",
				new Body());
		delegate.initAliases("test");
		OptionParser parser = delegate.getParser();
		includes = parser.accepts("includes").withRequiredArg().ofType(
				String.class);
		excludes = parser.accepts("excludes").withRequiredArg().ofType(
				String.class);
	}

	public Command getDelegate() {
		return delegate;
	}

	public OptionSpec<String> getIncludes() {
		return includes;
	}

	public ArgumentAcceptingOptionSpec<String> getExcludes() {
		return excludes;
	}

}
