/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.lang.internal;

import protoj.core.PropertyInfo;
import protoj.core.StandardProperties;
import protoj.lang.JunitFeature;
import protoj.lang.ScpFeature;
import protoj.lang.StandardProject;
import protoj.lang.UploadGoogleCodeFeature;
import protoj.lang.internal.ant.JunitTask;
import protoj.lang.internal.ant.ScpTask;

/**
 * Implements standard configuration based on system properties for the whole of
 * protoj.
 * 
 * @author Ashley Williams
 * 
 */
public aspect StandardConfig {

	/**
	 * Checks for attempts to call the google upload functionality from a
	 * {@link UploadGoogleCodeFeature} instance. If the
	 * {@link StandardProperties#getGcskipProperty()} system property is not
	 * present then the upload proceeds as normal.
	 */
	void around(UploadGoogleCodeFeature feature) : call(* net.bluecow.googlecode.ant.GoogleCodeUploadTask.execute())
		&& cflow(this(feature)) {
		StandardProject project = feature.getProject();
		StandardProperties properties = project.getProperties();
		PropertyInfo gcskipProperty = properties.getGcskipProperty();
		Boolean skipUpload = gcskipProperty.getBooleanValue();
		if (skipUpload) {
			StringBuilder builder = new StringBuilder();
			builder.append(gcskipProperty.getKey());
			builder.append(" property is set to true, so skipping the google");
			builder.append(" code upload with the following properties:\n");
			builder.append(feature.toString());
			project.getLogger().info(builder.toString());
		} else {
			proceed(feature);
		}
	}

	/**
	 * Checks for attempts to call the scp functionality from a
	 * {@link ScpFeature} instance. If the
	 * {@link StandardProperties#getScpskipProperty()} system property is not
	 * present then the remote copy proceeds as normal.
	 */
	void around(ScpFeature feature) : call(* ScpTask.execute())
		&& cflow(this(feature)) {
		StandardProject project = feature.getProject();
		PropertyInfo scpskipProperty = project.getProperties()
				.getScpskipProperty();
		Boolean skipScp = scpskipProperty.getBooleanValue();
		if (skipScp) {
			StringBuilder builder = new StringBuilder();
			builder.append(scpskipProperty.getKey());
			builder.append(" property is set to true, so skipping the remote");
			builder.append(" copy with the following properties:\n");
			builder.append(feature.toString());
			project.getLogger().info(builder.toString());
		} else {
			proceed(feature);
		}
	}

	/**
	 * Implements the configuration for the
	 * {@link StandardProperties#getDebugConditionProperty()} system property.
	 * 
	 * @param command
	 * @param project
	 */
	before(JunitFeature feature, JunitTask task) : call(* execute(..)) && this(feature) && target(task) {
		StandardProject project = feature.getProject();
		StandardProperties properties = project.getProperties();

		// only check other properties if debugging is switched on
		if (properties.getDebugProperty().getBooleanValue()) {
			String condition = properties.getDebugConditionProperty()
					.getValue();
			int port = properties.getDebugPortProperty().getIntegerValue();
			boolean suspend = properties.getDebugSuspendProperty()
					.getBooleanValue();
			if (condition != null && condition.equals("junit-task")) {
				task.initDebug(port, suspend);
				String message = "debug mode configured for junit tasks at port "
						+ port;
				project.getLogger().info(message);
			}
		}
	}

}
