/**
 * Copyright 2009 Ashley Williams
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You may
 * obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package protoj.lang.internal.ant;

import java.io.File;

import org.apache.tools.ant.Project;
import org.apache.tools.ant.taskdefs.Javadoc;
import org.apache.tools.ant.types.DirSet;

import protoj.core.internal.AntTarget;

/**
 * A convenience class for creating javadocs. Use the constructors to specify
 * the minimal and most widely anticipated configuration and the
 * <code>initXXX</code> methods for the less common configuration options.
 * 
 * @author Ashley Williams
 * 
 */
public final class JavadocTask {
	public static final String ERROR = "error";

	private AntTarget target;

	private Javadoc javadoc;

	private DirSet packageSet;

	/**
	 * Accepts the information required for creating javadocs.
	 * 
	 * @param srcDir
	 * @param destDir
	 */
	public JavadocTask(File srcDir, File destDir) {
		target = new AntTarget("protoj-javadocs");
		destDir.mkdirs();

		javadoc = new Javadoc();
		target.addTask(javadoc);
		javadoc.setTaskName("javadoc");
		javadoc.setDestdir(destDir);
		packageSet = new DirSet();
		packageSet.setDir(srcDir);
		javadoc.addPackageset(packageSet);
	}

	/**
	 * Ensures that all packages are included in the javadoc execution and that
	 * default excludes are applied.
	 * 
	 * @param memory
	 *            the maximum amount of memory to allocate to the javadoc vm.
	 */
	public void initDefault(String memory) {
		packageSet.setDefaultexcludes(true);
		javadoc.setMaxmemory(memory);
		packageSet.setIncludes("**");
	}

	public void execute() {
		target.execute();
	}

	/**
	 * A reference to the package set added to the javadoc ant task.
	 * 
	 * @return
	 */
	public DirSet getPackageSet() {
		return packageSet;
	}

	public Javadoc getjavadoc() {
		return javadoc;
	}

	/**
	 * Enables logging to the specified log file at Project.MSG_INFO level.
	 * 
	 * @param logFile
	 */
	public void initLogging(File logFile) {
		target.initLogging(logFile, Project.MSG_INFO);
	}
}
