/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.util.concurrent;

import com.google.common.collect.ImmutableClassToInstanceMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.testing.NullPointerTester;
import com.google.common.util.concurrent.RateLimiter;
import com.google.common.util.concurrent.SuppressUnderAndroid;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Random;
import java.util.concurrent.TimeUnit;
import junit.framework.TestCase;
import org.easymock.EasyMock;
import org.mockito.Mockito;

public class RateLimiterTest
extends TestCase {
    private static final double EPSILON = 1.0E-8;
    private final FakeStopwatch stopwatch = new FakeStopwatch();
    private static final ImmutableSet<String> NOT_WORKING_ON_MOCKS = ImmutableSet.of((Object)"latestPermitAgeSec", (Object)"setRate", (Object)"getAvailablePermits");
    private static final ImmutableClassToInstanceMap<Object> PARAMETER_VALUES = ImmutableClassToInstanceMap.builder().put(Integer.TYPE, (Object)1).put(Long.TYPE, (Object)1L).put(Double.TYPE, (Object)1.0).put(TimeUnit.class, (Object)TimeUnit.SECONDS).build();

    public void testSimple() {
        RateLimiter limiter = RateLimiter.create((RateLimiter.SleepingStopwatch)this.stopwatch, (double)5.0);
        limiter.acquire();
        limiter.acquire();
        limiter.acquire();
        this.assertEvents("R0.00", "R0.20", "R0.20");
    }

    public void testImmediateTryAcquire() {
        RateLimiter r = RateLimiter.create((double)1.0);
        RateLimiterTest.assertTrue((String)"Unable to acquire initial permit", (boolean)r.tryAcquire());
        RateLimiterTest.assertFalse((String)"Capable of acquiring secondary permit", (boolean)r.tryAcquire());
    }

    public void testDoubleMinValueCanAcquireExactlyOnce() {
        RateLimiter r = RateLimiter.create((RateLimiter.SleepingStopwatch)this.stopwatch, (double)Double.MIN_VALUE);
        RateLimiterTest.assertTrue((String)"Unable to acquire initial permit", (boolean)r.tryAcquire());
        RateLimiterTest.assertFalse((String)"Capable of acquiring an additional permit", (boolean)r.tryAcquire());
        this.stopwatch.sleepMillis(Integer.MAX_VALUE);
        RateLimiterTest.assertFalse((String)"Capable of acquiring an additional permit after sleeping", (boolean)r.tryAcquire());
    }

    public void testSimpleRateUpdate() {
        RateLimiter limiter = RateLimiter.create((double)5.0, (long)5L, (TimeUnit)TimeUnit.SECONDS);
        RateLimiterTest.assertEquals((Object)5.0, (Object)limiter.getRate());
        limiter.setRate(10.0);
        RateLimiterTest.assertEquals((Object)10.0, (Object)limiter.getRate());
        try {
            limiter.setRate(0.0);
            RateLimiterTest.fail();
        }
        catch (IllegalArgumentException expected) {
            // empty catch block
        }
        try {
            limiter.setRate(-10.0);
            RateLimiterTest.fail();
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    public void testAcquireParameterValidation() {
        RateLimiter limiter = RateLimiter.create((double)999.0);
        try {
            limiter.acquire(0);
            RateLimiterTest.fail();
        }
        catch (IllegalArgumentException expected) {
            // empty catch block
        }
        try {
            limiter.acquire(-1);
            RateLimiterTest.fail();
        }
        catch (IllegalArgumentException expected) {
            // empty catch block
        }
        try {
            limiter.tryAcquire(0);
            RateLimiterTest.fail();
        }
        catch (IllegalArgumentException expected) {
            // empty catch block
        }
        try {
            limiter.tryAcquire(-1);
            RateLimiterTest.fail();
        }
        catch (IllegalArgumentException expected) {
            // empty catch block
        }
        try {
            limiter.tryAcquire(0, 1L, TimeUnit.SECONDS);
            RateLimiterTest.fail();
        }
        catch (IllegalArgumentException expected) {
            // empty catch block
        }
        try {
            limiter.tryAcquire(-1, 1L, TimeUnit.SECONDS);
            RateLimiterTest.fail();
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    public void testSimpleWithWait() {
        RateLimiter limiter = RateLimiter.create((RateLimiter.SleepingStopwatch)this.stopwatch, (double)5.0);
        limiter.acquire();
        this.stopwatch.sleepMillis(200);
        limiter.acquire();
        limiter.acquire();
        this.assertEvents("R0.00", "U0.20", "R0.00", "R0.20");
    }

    public void testSimpleAcquireReturnValues() {
        RateLimiter limiter = RateLimiter.create((RateLimiter.SleepingStopwatch)this.stopwatch, (double)5.0);
        RateLimiterTest.assertEquals((double)0.0, (double)limiter.acquire(), (double)1.0E-8);
        this.stopwatch.sleepMillis(200);
        RateLimiterTest.assertEquals((double)0.0, (double)limiter.acquire(), (double)1.0E-8);
        RateLimiterTest.assertEquals((double)0.2, (double)limiter.acquire(), (double)1.0E-8);
        this.assertEvents("R0.00", "U0.20", "R0.00", "R0.20");
    }

    public void testSimpleAcquireEarliestAvailableIsInPast() {
        RateLimiter limiter = RateLimiter.create((RateLimiter.SleepingStopwatch)this.stopwatch, (double)5.0);
        RateLimiterTest.assertEquals((double)0.0, (double)limiter.acquire(), (double)1.0E-8);
        this.stopwatch.sleepMillis(400);
        RateLimiterTest.assertEquals((double)0.0, (double)limiter.acquire(), (double)1.0E-8);
        RateLimiterTest.assertEquals((double)0.0, (double)limiter.acquire(), (double)1.0E-8);
        RateLimiterTest.assertEquals((double)0.2, (double)limiter.acquire(), (double)1.0E-8);
    }

    public void testOneSecondBurst() {
        RateLimiter limiter = RateLimiter.create((RateLimiter.SleepingStopwatch)this.stopwatch, (double)5.0);
        this.stopwatch.sleepMillis(1000);
        this.stopwatch.sleepMillis(1000);
        limiter.acquire(1);
        limiter.acquire(1);
        limiter.acquire(3);
        limiter.acquire(1);
        limiter.acquire();
        this.assertEvents("U1.00", "U1.00", "R0.00", "R0.00", "R0.00", "R0.00", "R0.20");
    }

    public void testCreateWarmupParameterValidation() {
        RateLimiter.create((double)1.0, (long)1L, (TimeUnit)TimeUnit.NANOSECONDS);
        RateLimiter.create((double)1.0, (long)0L, (TimeUnit)TimeUnit.NANOSECONDS);
        try {
            RateLimiter.create((double)0.0, (long)1L, (TimeUnit)TimeUnit.NANOSECONDS);
            RateLimiterTest.fail();
        }
        catch (IllegalArgumentException expected) {
            // empty catch block
        }
        try {
            RateLimiter.create((double)1.0, (long)-1L, (TimeUnit)TimeUnit.NANOSECONDS);
            RateLimiterTest.fail();
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    @SuppressUnderAndroid
    public void testWarmUp() {
        int i;
        RateLimiter limiter = RateLimiter.create((RateLimiter.SleepingStopwatch)this.stopwatch, (double)2.0, (long)4000L, (TimeUnit)TimeUnit.MILLISECONDS, (double)3.0);
        for (i = 0; i < 8; ++i) {
            limiter.acquire();
        }
        this.stopwatch.sleepMillis(500);
        this.stopwatch.sleepMillis(4000);
        for (i = 0; i < 8; ++i) {
            limiter.acquire();
        }
        this.stopwatch.sleepMillis(500);
        this.stopwatch.sleepMillis(2000);
        for (i = 0; i < 8; ++i) {
            limiter.acquire();
        }
        this.assertEvents("R0.00, R1.38, R1.13, R0.88, R0.63, R0.50, R0.50, R0.50", "U0.50", "U4.00", "R0.00, R1.38, R1.13, R0.88, R0.63, R0.50, R0.50, R0.50", "U0.50", "U2.00", "R0.00, R0.50, R0.50, R0.50, R0.50, R0.50, R0.50, R0.50");
    }

    public void testWarmUpWithColdFactor() {
        int i;
        RateLimiter limiter = RateLimiter.create((RateLimiter.SleepingStopwatch)this.stopwatch, (double)5.0, (long)4000L, (TimeUnit)TimeUnit.MILLISECONDS, (double)10.0);
        for (i = 0; i < 8; ++i) {
            limiter.acquire();
        }
        this.stopwatch.sleepMillis(200);
        this.stopwatch.sleepMillis(4000);
        for (i = 0; i < 8; ++i) {
            limiter.acquire();
        }
        this.stopwatch.sleepMillis(200);
        this.stopwatch.sleepMillis(1000);
        for (i = 0; i < 8; ++i) {
            limiter.acquire();
        }
        this.assertEvents("R0.00, R1.75, R1.26, R0.76, R0.30, R0.20, R0.20, R0.20", "U0.20", "U4.00", "R0.00, R1.75, R1.26, R0.76, R0.30, R0.20, R0.20, R0.20", "U0.20", "U1.00", "R0.00, R0.20, R0.20, R0.20, R0.20, R0.20, R0.20, R0.20");
    }

    public void testWarmUpWithColdFactor1() {
        int i;
        RateLimiter limiter = RateLimiter.create((RateLimiter.SleepingStopwatch)this.stopwatch, (double)5.0, (long)4000L, (TimeUnit)TimeUnit.MILLISECONDS, (double)1.0);
        for (i = 0; i < 8; ++i) {
            limiter.acquire();
        }
        this.stopwatch.sleepMillis(340);
        for (i = 0; i < 8; ++i) {
            limiter.acquire();
        }
        this.assertEvents("R0.00, R0.20, R0.20, R0.20, R0.20, R0.20, R0.20, R0.20", "U0.34", "R0.00, R0.20, R0.20, R0.20, R0.20, R0.20, R0.20, R0.20");
    }

    @SuppressUnderAndroid
    public void testWarmUpAndUpdate() {
        int i;
        RateLimiter limiter = RateLimiter.create((RateLimiter.SleepingStopwatch)this.stopwatch, (double)2.0, (long)4000L, (TimeUnit)TimeUnit.MILLISECONDS, (double)3.0);
        for (i = 0; i < 8; ++i) {
            limiter.acquire();
        }
        this.stopwatch.sleepMillis(4500);
        for (i = 0; i < 3; ++i) {
            limiter.acquire();
        }
        limiter.setRate(4.0);
        limiter.acquire();
        for (i = 0; i < 4; ++i) {
            limiter.acquire();
        }
        this.stopwatch.sleepMillis(4250);
        for (i = 0; i < 11; ++i) {
            limiter.acquire();
        }
        this.assertEvents("R0.00, R1.38, R1.13, R0.88, R0.63, R0.50, R0.50, R0.50", "U4.50", "R0.00, R1.38, R1.13", "R0.88", "R0.34, R0.28, R0.25, R0.25", "U4.25", "R0.00, R0.72, R0.66, R0.59, R0.53, R0.47, R0.41", "R0.34, R0.28, R0.25, R0.25");
    }

    public void testWarmUpAndUpdateWithColdFactor() {
        int i;
        RateLimiter limiter = RateLimiter.create((RateLimiter.SleepingStopwatch)this.stopwatch, (double)5.0, (long)4000L, (TimeUnit)TimeUnit.MILLISECONDS, (double)10.0);
        for (i = 0; i < 8; ++i) {
            limiter.acquire();
        }
        this.stopwatch.sleepMillis(4200);
        for (i = 0; i < 3; ++i) {
            limiter.acquire();
        }
        limiter.setRate(10.0);
        limiter.acquire();
        for (i = 0; i < 4; ++i) {
            limiter.acquire();
        }
        this.stopwatch.sleepMillis(4100);
        for (i = 0; i < 11; ++i) {
            limiter.acquire();
        }
        this.assertEvents("R0.00, R1.75, R1.26, R0.76, R0.30, R0.20, R0.20, R0.20", "U4.20", "R0.00, R1.75, R1.26", "R0.76", "R0.20, R0.10, R0.10, R0.10", "U4.10", "R0.00, R0.94, R0.81, R0.69, R0.57, R0.44, R0.32", "R0.20, R0.10, R0.10, R0.10");
    }

    public void testBurstyAndUpdate() {
        RateLimiter rateLimiter = RateLimiter.create((RateLimiter.SleepingStopwatch)this.stopwatch, (double)1.0);
        rateLimiter.acquire(1);
        rateLimiter.acquire(1);
        rateLimiter.setRate(2.0);
        rateLimiter.acquire(1);
        rateLimiter.acquire(2);
        rateLimiter.acquire(4);
        rateLimiter.acquire(1);
        this.assertEvents("R0.00", "R1.00", "R1.00", "R0.50", "R1.00", "R2.00");
    }

    public void testTryAcquire_noWaitAllowed() {
        RateLimiter limiter = RateLimiter.create((RateLimiter.SleepingStopwatch)this.stopwatch, (double)5.0);
        RateLimiterTest.assertTrue((boolean)limiter.tryAcquire(0L, TimeUnit.SECONDS));
        RateLimiterTest.assertFalse((boolean)limiter.tryAcquire(0L, TimeUnit.SECONDS));
        RateLimiterTest.assertFalse((boolean)limiter.tryAcquire(0L, TimeUnit.SECONDS));
        this.stopwatch.sleepMillis(100);
        RateLimiterTest.assertFalse((boolean)limiter.tryAcquire(0L, TimeUnit.SECONDS));
    }

    public void testTryAcquire_someWaitAllowed() {
        RateLimiter limiter = RateLimiter.create((RateLimiter.SleepingStopwatch)this.stopwatch, (double)5.0);
        RateLimiterTest.assertTrue((boolean)limiter.tryAcquire(0L, TimeUnit.SECONDS));
        RateLimiterTest.assertTrue((boolean)limiter.tryAcquire(200L, TimeUnit.MILLISECONDS));
        RateLimiterTest.assertFalse((boolean)limiter.tryAcquire(100L, TimeUnit.MILLISECONDS));
        this.stopwatch.sleepMillis(100);
        RateLimiterTest.assertTrue((boolean)limiter.tryAcquire(100L, TimeUnit.MILLISECONDS));
    }

    public void testTryAcquire_overflow() {
        RateLimiter limiter = RateLimiter.create((RateLimiter.SleepingStopwatch)this.stopwatch, (double)5.0);
        RateLimiterTest.assertTrue((boolean)limiter.tryAcquire(0L, TimeUnit.MICROSECONDS));
        this.stopwatch.sleepMillis(100);
        RateLimiterTest.assertTrue((boolean)limiter.tryAcquire(Long.MAX_VALUE, TimeUnit.MICROSECONDS));
    }

    public void testTryAcquire_negative() {
        RateLimiter limiter = RateLimiter.create((RateLimiter.SleepingStopwatch)this.stopwatch, (double)5.0);
        RateLimiterTest.assertTrue((boolean)limiter.tryAcquire(5, 0L, TimeUnit.SECONDS));
        this.stopwatch.sleepMillis(900);
        RateLimiterTest.assertFalse((boolean)limiter.tryAcquire(1, Long.MIN_VALUE, TimeUnit.SECONDS));
        this.stopwatch.sleepMillis(100);
        RateLimiterTest.assertTrue((boolean)limiter.tryAcquire(1, -1L, TimeUnit.SECONDS));
    }

    public void testSimpleWeights() {
        RateLimiter rateLimiter = RateLimiter.create((RateLimiter.SleepingStopwatch)this.stopwatch, (double)1.0);
        rateLimiter.acquire(1);
        rateLimiter.acquire(1);
        rateLimiter.acquire(2);
        rateLimiter.acquire(4);
        rateLimiter.acquire(8);
        rateLimiter.acquire(1);
        this.assertEvents("R0.00", "R1.00", "R1.00", "R2.00", "R4.00", "R8.00");
    }

    public void testInfinity_Bursty() {
        RateLimiter limiter = RateLimiter.create((RateLimiter.SleepingStopwatch)this.stopwatch, (double)Double.POSITIVE_INFINITY);
        limiter.acquire(0x1FFFFFFF);
        limiter.acquire(0x3FFFFFFF);
        limiter.acquire(Integer.MAX_VALUE);
        this.assertEvents("R0.00", "R0.00", "R0.00");
        limiter.setRate(2.0);
        limiter.acquire();
        limiter.acquire();
        limiter.acquire();
        limiter.acquire();
        limiter.acquire();
        this.assertEvents("R0.00", "R0.00", "R0.00", "R0.50", "R0.50");
        limiter.setRate(Double.POSITIVE_INFINITY);
        limiter.acquire();
        limiter.acquire();
        limiter.acquire();
        this.assertEvents("R0.50", "R0.00", "R0.00");
    }

    public void testInfinity_BustyTimeElapsed() {
        RateLimiter limiter = RateLimiter.create((RateLimiter.SleepingStopwatch)this.stopwatch, (double)Double.POSITIVE_INFINITY);
        this.stopwatch.instant += 1000000L;
        limiter.setRate(2.0);
        for (int i = 0; i < 5; ++i) {
            limiter.acquire();
        }
        this.assertEvents("R0.00", "R0.00", "R0.00", "R0.50", "R0.50");
    }

    public void testInfinity_WarmUp() {
        RateLimiter limiter = RateLimiter.create((RateLimiter.SleepingStopwatch)this.stopwatch, (double)Double.POSITIVE_INFINITY, (long)10L, (TimeUnit)TimeUnit.SECONDS, (double)3.0);
        limiter.acquire(0x1FFFFFFF);
        limiter.acquire(0x3FFFFFFF);
        limiter.acquire(Integer.MAX_VALUE);
        this.assertEvents("R0.00", "R0.00", "R0.00");
        limiter.setRate(1.0);
        limiter.acquire();
        limiter.acquire();
        limiter.acquire();
        this.assertEvents("R0.00", "R1.00", "R1.00");
        limiter.setRate(Double.POSITIVE_INFINITY);
        limiter.acquire();
        limiter.acquire();
        limiter.acquire();
        this.assertEvents("R1.00", "R0.00", "R0.00");
    }

    public void testInfinity_WarmUpTimeElapsed() {
        RateLimiter limiter = RateLimiter.create((RateLimiter.SleepingStopwatch)this.stopwatch, (double)Double.POSITIVE_INFINITY, (long)10L, (TimeUnit)TimeUnit.SECONDS, (double)3.0);
        this.stopwatch.instant += 1000000L;
        limiter.setRate(1.0);
        for (int i = 0; i < 5; ++i) {
            limiter.acquire();
        }
        this.assertEvents("R0.00", "R1.00", "R1.00", "R1.00", "R1.00");
    }

    public void testWeNeverGetABurstMoreThanOneSec() {
        int[] rates;
        RateLimiter limiter = RateLimiter.create((RateLimiter.SleepingStopwatch)this.stopwatch, (double)1.0);
        int[] arr$ = rates = new int[]{1000, 1, 10, 1000000, 10, 1};
        int len$ = arr$.length;
        for (int i$ = 0; i$ < len$; ++i$) {
            int rate;
            int oneSecWorthOfWork = rate = arr$[i$];
            this.stopwatch.sleepMillis(rate * 1000);
            limiter.setRate((double)rate);
            long burst = this.measureTotalTimeMillis(limiter, oneSecWorthOfWork, new Random());
            RateLimiterTest.assertTrue((burst <= 1000L ? 1 : 0) != 0);
            long afterBurst = this.measureTotalTimeMillis(limiter, oneSecWorthOfWork, new Random());
            RateLimiterTest.assertTrue((afterBurst >= 1000L ? 1 : 0) != 0);
        }
    }

    public void testTimeToWarmUpIsHonouredEvenWithWeights() {
        Random random = new Random();
        int warmupPermits = 10;
        double[] coldFactorsToTest = new double[]{2.0, 3.0, 10.0};
        double[] qpsToTest = new double[]{4.0, 2.0, 1.0, 0.5, 0.1};
        for (int trial = 0; trial < 100; ++trial) {
            for (double coldFactor : coldFactorsToTest) {
                for (double qps : qpsToTest) {
                    long warmupMillis = (long)((1.0 + coldFactor) * (double)warmupPermits / (2.0 * qps) * 1000.0);
                    RateLimiter rateLimiter = RateLimiter.create((RateLimiter.SleepingStopwatch)this.stopwatch, (double)qps, (long)warmupMillis, (TimeUnit)TimeUnit.MILLISECONDS, (double)coldFactor);
                    RateLimiterTest.assertEquals((long)warmupMillis, (long)this.measureTotalTimeMillis(rateLimiter, warmupPermits, random));
                }
            }
        }
    }

    public void testNulls() {
        NullPointerTester tester = new NullPointerTester().setDefault(RateLimiter.SleepingStopwatch.class, (Object)this.stopwatch).setDefault(Integer.TYPE, (Object)1).setDefault(Double.TYPE, (Object)1.0);
        tester.testStaticMethods(RateLimiter.class, NullPointerTester.Visibility.PACKAGE);
        tester.testInstanceMethods((Object)RateLimiter.create((RateLimiter.SleepingStopwatch)this.stopwatch, (double)5.0), NullPointerTester.Visibility.PACKAGE);
    }

    private long measureTotalTimeMillis(RateLimiter rateLimiter, int permits, Random random) {
        long startTime = this.stopwatch.instant;
        while (permits > 0) {
            int nextPermitsToAcquire = Math.max(1, random.nextInt(permits));
            permits -= nextPermitsToAcquire;
            rateLimiter.acquire(nextPermitsToAcquire);
        }
        rateLimiter.acquire(1);
        return TimeUnit.NANOSECONDS.toMillis(this.stopwatch.instant - startTime);
    }

    private void assertEvents(String ... events) {
        RateLimiterTest.assertEquals((String)Arrays.toString(events), (String)this.stopwatch.readEventsAndClear());
    }

    public void testMockingMockito() throws Exception {
        RateLimiter mock = (RateLimiter)Mockito.mock(RateLimiter.class);
        RateLimiterTest.doTestMocking(mock);
    }

    @SuppressUnderAndroid
    public void testMockingEasyMock() throws Exception {
        RateLimiter mock = (RateLimiter)EasyMock.createNiceMock(RateLimiter.class);
        EasyMock.replay((Object[])new Object[]{mock});
        RateLimiterTest.doTestMocking(mock);
    }

    private static void doTestMocking(RateLimiter mock) throws Exception {
        for (Method method : RateLimiter.class.getMethods()) {
            if (Modifier.isStatic(method.getModifiers()) || NOT_WORKING_ON_MOCKS.contains((Object)method.getName()) || method.getDeclaringClass().equals(Object.class)) continue;
            method.invoke((Object)mock, RateLimiterTest.arbitraryParameters(method));
        }
    }

    private static Object[] arbitraryParameters(Method method) {
        Class<?>[] parameterTypes = method.getParameterTypes();
        Object[] params = new Object[parameterTypes.length];
        for (int i = 0; i < parameterTypes.length; ++i) {
            params[i] = PARAMETER_VALUES.get(parameterTypes[i]);
        }
        return params;
    }

    static class FakeStopwatch
    extends RateLimiter.SleepingStopwatch {
        long instant = 0L;
        final List<String> events = Lists.newArrayList();

        FakeStopwatch() {
        }

        public long readMicros() {
            return TimeUnit.NANOSECONDS.toMicros(this.instant);
        }

        void sleepMillis(int millis) {
            this.sleepMicros("U", TimeUnit.MILLISECONDS.toMicros(millis));
        }

        void sleepMicros(String caption, long micros) {
            this.instant += TimeUnit.MICROSECONDS.toNanos(micros);
            this.events.add(caption + String.format(Locale.ROOT, "%3.2f", (double)micros / 1000000.0));
        }

        void sleepMicrosUninterruptibly(long micros) {
            this.sleepMicros("R", micros);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        String readEventsAndClear() {
            try {
                String string = this.events.toString();
                return string;
            }
            finally {
                this.events.clear();
            }
        }

        public String toString() {
            return this.events.toString();
        }
    }
}

