/**
 * Copyright 2012 Niall Gallagher
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.googlecode.cqengine.index.standingquery;

import com.googlecode.cqengine.index.Index;
import com.googlecode.cqengine.query.Query;
import com.googlecode.cqengine.query.option.QueryOption;
import com.googlecode.cqengine.resultset.ResultSet;
import com.googlecode.cqengine.resultset.stored.StoredResultSet;
import com.googlecode.cqengine.resultset.stored.StoredSetBasedResultSet;

import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author Niall Gallagher
 */
public class StandingQueryIndex<O> implements Index<O> {

    private static final int INDEX_RETRIEVAL_COST = 10;

    private final Query<O> standingQuery;

    private final StoredResultSet<O> storedResultSet = new StoredSetBasedResultSet<O>(
            Collections.<O>newSetFromMap(new ConcurrentHashMap<O, Boolean>()),
            INDEX_RETRIEVAL_COST
    );

    public StandingQueryIndex(Query<O> standingQuery) {
        this.standingQuery = standingQuery;
    }

    /**
     * {@inheritDoc}
     * <p/>
     * This index is mutable.
     *
     * @return true
     */
    @Override
    public boolean isMutable() {
        return true;
    }

    public Query<O> getStandingQuery() {
        return standingQuery;
    }

    /**
     * {@inheritDoc}
     * <p/>
     * <b>This implementation always returns true, as this index supports all types of query.</b>
     *
     * @return true, this index supports all types of query
     */
    @Override
    public boolean supportsQuery(Query<O> query) {
        return standingQuery.equals(query);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public ResultSet<O> retrieve(final Query<O> query, Map<Class<? extends QueryOption>, QueryOption<O>> queryOptions) {
        if (!supportsQuery(query)) {
            // TODO: check necessary?
            throw new IllegalArgumentException("Unsupported query: " + query);
        }
        return storedResultSet;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void init(Set<O> collection) {
        storedResultSet.clear();
        notifyObjectsAdded(collection);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void notifyObjectsAdded(Collection<O> objects) {
        for (O object : objects) {
            if (standingQuery.matches(object)) {
                storedResultSet.add(object);
            }
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void notifyObjectsRemoved(Collection<O> objects) {
        for (O object : objects) {
            if (standingQuery.matches(object)) {
                storedResultSet.remove(object);
            }
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void notifyObjectsCleared() {
        storedResultSet.clear();
    }

    @Override
    public String toString() {
        return "StandingQueryIndex{" +
                "standingQuery=" + standingQuery +
                '}';
    }

    /**
     * A static factory method for convenience.
     * <p/>
     * Equivalent to {@code new StandingQueryIndex&lt;Query&lt;O&gt;, O&gt;(standingQuery)}.
     * <p/>
     * @param standingQuery The standing query on which the index will be built
     * @param <O> The type of the objects in the collection being indexed
     * @return A new StandingQueryIndex which will build an index on this standing query
     */
    public static <O> StandingQueryIndex<O> onQuery(Query<O> standingQuery) {
        return new StandingQueryIndex<O>(standingQuery);
    }
}
