package com.jsftoolkit.gen.info;

import com.jsftoolkit.utils.serial.XmlAttribute;
import com.jsftoolkit.utils.serial.XmlElement;

/**
 * Describes a property of a component, including whether it is required or not
 * and the default value.
 * 
 * @author noah
 * 
 */
@XmlElement(tagName = "property")
public class PropertyInfo {
	private Class<?> type;

	private String name;

	private String defaultValue;

	private boolean required;

	public PropertyInfo() {
		this(null, null, null);
	}

	/**
	 * Creates a {@link String} property with the given name.
	 * 
	 * @param name
	 */
	public PropertyInfo(final String name) {
		this(String.class, name);
	}

	/**
	 * Creates a property that defaults to null
	 * 
	 * @param _class
	 * @param name
	 */
	public PropertyInfo(final Class<?> _class, final String name) {
		this(_class, name, null);
	}

	/**
	 * Same as {@link #PropertyInfo(Class, String, String, boolean)}, but with
	 * required set to false.
	 * 
	 * @param _class
	 * @param name
	 * @param defaultValue
	 */
	public PropertyInfo(final Class<?> _class, final String name,
			final String defaultValue) {
		this(_class, name, defaultValue, false);
	}

	/**
	 * 
	 * @param _class
	 *            the class of the property (leave null for properties that
	 *            should only pass through a constant and be specified in the
	 *            tag handler, such as VALUE)
	 * @param name
	 *            the name of the property.
	 * @param defaultValue
	 *            the java code string to initialize this value. e.g. for the
	 *            int 3, "3". For the string 'foo' "\"Foo\"". For some arbitrary
	 *            object: "new MyClass(foo,bar,baz)";
	 * @param required
	 *            id this property required?
	 */
	public PropertyInfo(final Class<?> _class, final String name,
			final String defaultValue, final boolean required) {
		super();
		this.type = _class;
		this.name = name;
		this.defaultValue = defaultValue;
		this.required = required;
	}

	/**
	 * 
	 * @return the class of the property
	 */
	@XmlAttribute
	public Class<?> getType() {
		return type;
	}

	/**
	 * 
	 * @return the name of the property
	 */
	@XmlAttribute
	public String getName() {
		return name;
	}

	/**
	 * 
	 * @return the code to generate the default value. e.g. to set it to a 3
	 *         call setDefaultValue("3"). For the string foo:
	 *         setDefaultValue("\"foo\"")
	 */
	@XmlAttribute
	public String getDefaultValue() {
		return defaultValue;
	}

	/**
	 * 
	 * @return true if the attribute is required
	 */
	@XmlAttribute
	public boolean isRequired() {
		return required;
	}

	public void setType(Class<?> _class) {
		this.type = _class;
	}

	public void setDefaultValue(String defaultValue) {
		this.defaultValue = defaultValue;
	}

	public void setName(String name) {
		this.name = name;
	}

	public void setRequired(boolean required) {
		this.required = required;
	}

	@Override
	public int hashCode() {
		final int PRIME = 31;
		int result = 1;
		result = PRIME * result + ((name == null) ? 0 : name.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		final PropertyInfo other = (PropertyInfo) obj;
		if (name == null) {
			if (other.name != null)
				return false;
		} else if (!name.equals(other.name))
			return false;
		return true;
	}

	/**
	 * 
	 * @return the code to regenerate this {@link PropertyInfo} instance
	 */
	public CharSequence getCodeString() {
		return String.format("new %s(%s,\"%s\",%s)",
				getClass().getSimpleName(), type == null ? null : type
						.getSimpleName()
						+ ".class", name, defaultValue);
	}

}
