package com.jsftoolkit.utils;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.Charset;
import java.util.Map;

import javax.faces.context.FacesContext;
import javax.faces.context.ResponseWriter;

import org.apache.shale.remoting.Mechanism;
import org.apache.shale.remoting.XhtmlHelper;
import org.apache.shale.remoting.faces.ResponseFactory;

/**
 * Translates embedded references to classpath resources to proper remoting
 * URLs.
 * <p>
 * e.g. 'url(${resource:/com/yourCompany/foo.css})' gets translated to something
 * like 'url(/AppName/faces/static/com/yourCompany/foo.css)'
 * 
 * @author noah
 * 
 */
public class ResourceFilterImpl implements ResourceFilter {

	private static final XhtmlHelper HELPER = new XhtmlHelper();

	private String prefix = "${resource:";

	private String suffix = "}";

	private int bufferSize = 65536;

	/**
	 * Stateless factory for {@link ResponseWriter} instances.
	 */
	private static final ResponseFactory RESPONSE_FACTORY = new ResponseFactory();

	/**
	 * The resource is loaded and copied to the response, translating
	 * occurrences of a pattern enclosed in {@link #prefix} and {@link #suffix}
	 * with proper shale remoting URLs to the referenced resource.
	 * 
	 * @see com.jsftoolkit.utils.ResourceFilter#writeFilteredResource()
	 */
	public void writeFilteredResource() throws IOException {
		FacesContext context = FacesContext.getCurrentInstance();
		Map<String, String> requestMap = context.getExternalContext()
				.getRequestParameterMap();

		// get the parameters
		String resourceId = (String) requestMap.get(RESOURCE_KEY);
		String contentType = Utils.toString(requestMap.get(CONTENT_TYPE_KEY),
				"text/plain");
		String charsetName = Utils.toString(requestMap.get(CHARSET_KEY),
				"UTF-8");
		Charset charset = Utils.getCharset(charsetName);

		// create a response writer
		ResponseWriter out = RESPONSE_FACTORY.getResponseWriter(context,
				contentType);

		if (resourceId != null) {
			// load and filter the resource
			InputStream stream = getClass().getResourceAsStream(resourceId);
			if (stream != null) {
				// decorate the stream appropriately
				IsFinishedReader in = new IsFinishedReader(new BufferedReader(
						new InputStreamReader(stream, charset)));
				DelimitedReader reader;

				// temp buffer for copying to the response writer
				char[] buffer = new char[bufferSize];

				while (!in.isFinished()) {
					// copy up to a resource name, discarding the prefix
					reader = new DelimitedReader(in, prefix);
					Utils.writeAll(reader, out, buffer);

					if (!in.isFinished()) {
						// read the resource path
						String resource = Utils.toString(new DelimitedReader(
								reader.getReader(), suffix));

						// write the real path to the resource
						String url = HELPER.mapResourceId(context,
								Mechanism.CLASS_RESOURCE, resource);
						out.write(url);
					}
				}
			} 
		} 

		out.close();
	}

	/**
	 * 
	 * @return the prefix of resource references e.g. '${'
	 */
	public String getPrefix() {
		return prefix;
	}

	public void setPrefix(String prefix) {
		this.prefix = prefix;
	}

	/**
	 * 
	 * @return the suffix of resource references, e.g. '}'
	 */
	public String getSuffix() {
		return suffix;
	}

	public void setSuffix(String suffix) {
		this.suffix = suffix;
	}

	/**
	 * 
	 * @return the size of the buffer to use while reading the resource.
	 */
	public int getBufferSize() {
		return bufferSize;
	}

	public void setBufferSize(int bufferSize) {
		this.bufferSize = bufferSize;
	}

}
