package upickle.core


/**
  * A fast buffer that can be used to store Bytes (Bytes or Chars).
  *
  * Generally faster than the equivalent [[StringBuilder]] or
  * [[java.io.ByteArrayOutputStream]], since:
  *
  * - It is specialized and without the overhead of polymorphism or synchronization.
  * - It allows the user to call `ensureLength` and `appendUnsafe` separately, e.g.
  *   callign `ensureLength` once before `appendUnsafe`-ing multiple Bytes
  * - It provides fast methods like [[makeString]] or [[writeOutToIfLongerThan]], that
  *   let you push the data elsewhere with minimal unnecessary copying
  */
class ByteBuilder(startSize: Int = 32) extends upickle.core.ByteAppendC{
  var arr: Array[Byte] = new Array(startSize)
  var length: Int = 0
  private def getArr = arr
  def getLength = length
  def reset(): Unit = length = 0
  def ensureLength(increment: Int): Unit = {
    var multiple = arr.length
    val targetLength = length + increment
    while (multiple < targetLength) multiple = multiple * 2
    if (multiple != arr.length) arr = java.util.Arrays.copyOf(arr, multiple)
  }
  def append(x: Int): Unit = append(x.toByte)
  def append(x: Byte): Unit = {
    if (length == arr.length) arr = java.util.Arrays.copyOf(arr, arr.length * 2)
    arr(length) = x
    length += 1
  }
  def appendAll(bytes: Array[Byte], bytesLength: Int): Unit = appendAll(bytes, 0, bytesLength)

  def appendAll(bytes: Array[Byte], bytesStart: Int, bytesLength: Int): Unit = {
    ensureLength(bytesLength)
    System.arraycopy(bytes, bytesStart, arr, length, bytesLength)
    length += bytesLength
  }
  def appendAllUnsafe(other: ByteBuilder): Unit = {
    val bytesLength = other.getLength
    System.arraycopy(other.getArr, 0, arr, length, bytesLength)
    length += bytesLength
  }

  def appendUnsafeC(x: Char): Unit = appendUnsafe(x.toByte)
  def appendUnsafe(x: Byte): Unit = {
    arr(length) = x
    length += 1
  }

  def makeString(): String = ByteOps.newString(arr, 0, length)
  def writeOutToIfLongerThan(writer: upickle.core.ByteOps.Output, threshold: Int): Unit = {
    if (length > threshold) {
      writer.write(arr, 0, length)
      length = 0
    }
  }

}
