/*
 * Copyright (c) Microsoft Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.microsoft.playwright;

import com.microsoft.playwright.options.*;
import java.nio.file.Path;
import java.util.*;

/**
 * ElementHandle represents an in-page DOM element. ElementHandles can be created with the {@link Page#querySelector
 * Page.querySelector()} method.
 * <pre>{@code
 * import com.microsoft.playwright.*;
 *
 * public class Example {
 *   public static void main(String[] args) {
 *     try (Playwright playwright = Playwright.create()) {
 *       BrowserType chromium = playwright.chromium();
 *       Browser browser = chromium.launch();
 *       Page page = browser.newPage();
 *       page.navigate("https://example.com");
 *       ElementHandle hrefElement = page.querySelector("a");
 *       hrefElement.click();
 *       // ...
 *     }
 *   }
 * }
 * }</pre>
 *
 * <p> ElementHandle prevents DOM element from garbage collection unless the handle is disposed with {@link JSHandle#dispose
 * JSHandle.dispose()}. ElementHandles are auto-disposed when their origin frame gets navigated.
 *
 * <p> ElementHandle instances can be used as an argument in {@link Page#evalOnSelector Page.evalOnSelector()} and {@link
 * Page#evaluate Page.evaluate()} methods.
 */
public interface ElementHandle extends JSHandle {
  class CheckOptions {
    /**
     * Whether to bypass the <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks. Defaults to
     * {@code false}.
     */
    public Boolean force;
    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to {@code false}.
     */
    public Boolean noWaitAfter;
    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of the
     * element.
     */
    public Position position;
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass {@code 0} to disable timeout. The default value can be changed by
     * using the {@link BrowserContext#setDefaultTimeout BrowserContext.setDefaultTimeout()} or {@link Page#setDefaultTimeout
     * Page.setDefaultTimeout()} methods.
     */
    public Double timeout;
    /**
     * When set, this method only performs the <a href="https://playwright.dev/java/docs/actionability/">actionability</a>
     * checks and skips the action. Defaults to {@code false}. Useful to wait until the element is ready for the action without
     * performing it.
     */
    public Boolean trial;

    public CheckOptions setForce(boolean force) {
      this.force = force;
      return this;
    }
    public CheckOptions setNoWaitAfter(boolean noWaitAfter) {
      this.noWaitAfter = noWaitAfter;
      return this;
    }
    public CheckOptions setPosition(double x, double y) {
      return setPosition(new Position(x, y));
    }
    public CheckOptions setPosition(Position position) {
      this.position = position;
      return this;
    }
    public CheckOptions setTimeout(double timeout) {
      this.timeout = timeout;
      return this;
    }
    public CheckOptions setTrial(boolean trial) {
      this.trial = trial;
      return this;
    }
  }
  class ClickOptions {
    /**
     * Defaults to {@code left}.
     */
    public MouseButton button;
    /**
     * defaults to 1. See [UIEvent.detail].
     */
    public Integer clickCount;
    /**
     * Time to wait between {@code mousedown} and {@code mouseup} in milliseconds. Defaults to 0.
     */
    public Double delay;
    /**
     * Whether to bypass the <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks. Defaults to
     * {@code false}.
     */
    public Boolean force;
    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores current
     * modifiers back. If not specified, currently pressed modifiers are used.
     */
    public List<KeyboardModifier> modifiers;
    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to {@code false}.
     */
    public Boolean noWaitAfter;
    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of the
     * element.
     */
    public Position position;
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass {@code 0} to disable timeout. The default value can be changed by
     * using the {@link BrowserContext#setDefaultTimeout BrowserContext.setDefaultTimeout()} or {@link Page#setDefaultTimeout
     * Page.setDefaultTimeout()} methods.
     */
    public Double timeout;
    /**
     * When set, this method only performs the <a href="https://playwright.dev/java/docs/actionability/">actionability</a>
     * checks and skips the action. Defaults to {@code false}. Useful to wait until the element is ready for the action without
     * performing it.
     */
    public Boolean trial;

    public ClickOptions setButton(MouseButton button) {
      this.button = button;
      return this;
    }
    public ClickOptions setClickCount(int clickCount) {
      this.clickCount = clickCount;
      return this;
    }
    public ClickOptions setDelay(double delay) {
      this.delay = delay;
      return this;
    }
    public ClickOptions setForce(boolean force) {
      this.force = force;
      return this;
    }
    public ClickOptions setModifiers(List<KeyboardModifier> modifiers) {
      this.modifiers = modifiers;
      return this;
    }
    public ClickOptions setNoWaitAfter(boolean noWaitAfter) {
      this.noWaitAfter = noWaitAfter;
      return this;
    }
    public ClickOptions setPosition(double x, double y) {
      return setPosition(new Position(x, y));
    }
    public ClickOptions setPosition(Position position) {
      this.position = position;
      return this;
    }
    public ClickOptions setTimeout(double timeout) {
      this.timeout = timeout;
      return this;
    }
    public ClickOptions setTrial(boolean trial) {
      this.trial = trial;
      return this;
    }
  }
  class DblclickOptions {
    /**
     * Defaults to {@code left}.
     */
    public MouseButton button;
    /**
     * Time to wait between {@code mousedown} and {@code mouseup} in milliseconds. Defaults to 0.
     */
    public Double delay;
    /**
     * Whether to bypass the <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks. Defaults to
     * {@code false}.
     */
    public Boolean force;
    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores current
     * modifiers back. If not specified, currently pressed modifiers are used.
     */
    public List<KeyboardModifier> modifiers;
    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to {@code false}.
     */
    public Boolean noWaitAfter;
    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of the
     * element.
     */
    public Position position;
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass {@code 0} to disable timeout. The default value can be changed by
     * using the {@link BrowserContext#setDefaultTimeout BrowserContext.setDefaultTimeout()} or {@link Page#setDefaultTimeout
     * Page.setDefaultTimeout()} methods.
     */
    public Double timeout;
    /**
     * When set, this method only performs the <a href="https://playwright.dev/java/docs/actionability/">actionability</a>
     * checks and skips the action. Defaults to {@code false}. Useful to wait until the element is ready for the action without
     * performing it.
     */
    public Boolean trial;

    public DblclickOptions setButton(MouseButton button) {
      this.button = button;
      return this;
    }
    public DblclickOptions setDelay(double delay) {
      this.delay = delay;
      return this;
    }
    public DblclickOptions setForce(boolean force) {
      this.force = force;
      return this;
    }
    public DblclickOptions setModifiers(List<KeyboardModifier> modifiers) {
      this.modifiers = modifiers;
      return this;
    }
    public DblclickOptions setNoWaitAfter(boolean noWaitAfter) {
      this.noWaitAfter = noWaitAfter;
      return this;
    }
    public DblclickOptions setPosition(double x, double y) {
      return setPosition(new Position(x, y));
    }
    public DblclickOptions setPosition(Position position) {
      this.position = position;
      return this;
    }
    public DblclickOptions setTimeout(double timeout) {
      this.timeout = timeout;
      return this;
    }
    public DblclickOptions setTrial(boolean trial) {
      this.trial = trial;
      return this;
    }
  }
  class FillOptions {
    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to {@code false}.
     */
    public Boolean noWaitAfter;
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass {@code 0} to disable timeout. The default value can be changed by
     * using the {@link BrowserContext#setDefaultTimeout BrowserContext.setDefaultTimeout()} or {@link Page#setDefaultTimeout
     * Page.setDefaultTimeout()} methods.
     */
    public Double timeout;

    public FillOptions setNoWaitAfter(boolean noWaitAfter) {
      this.noWaitAfter = noWaitAfter;
      return this;
    }
    public FillOptions setTimeout(double timeout) {
      this.timeout = timeout;
      return this;
    }
  }
  class HoverOptions {
    /**
     * Whether to bypass the <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks. Defaults to
     * {@code false}.
     */
    public Boolean force;
    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores current
     * modifiers back. If not specified, currently pressed modifiers are used.
     */
    public List<KeyboardModifier> modifiers;
    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of the
     * element.
     */
    public Position position;
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass {@code 0} to disable timeout. The default value can be changed by
     * using the {@link BrowserContext#setDefaultTimeout BrowserContext.setDefaultTimeout()} or {@link Page#setDefaultTimeout
     * Page.setDefaultTimeout()} methods.
     */
    public Double timeout;
    /**
     * When set, this method only performs the <a href="https://playwright.dev/java/docs/actionability/">actionability</a>
     * checks and skips the action. Defaults to {@code false}. Useful to wait until the element is ready for the action without
     * performing it.
     */
    public Boolean trial;

    public HoverOptions setForce(boolean force) {
      this.force = force;
      return this;
    }
    public HoverOptions setModifiers(List<KeyboardModifier> modifiers) {
      this.modifiers = modifiers;
      return this;
    }
    public HoverOptions setPosition(double x, double y) {
      return setPosition(new Position(x, y));
    }
    public HoverOptions setPosition(Position position) {
      this.position = position;
      return this;
    }
    public HoverOptions setTimeout(double timeout) {
      this.timeout = timeout;
      return this;
    }
    public HoverOptions setTrial(boolean trial) {
      this.trial = trial;
      return this;
    }
  }
  class PressOptions {
    /**
     * Time to wait between {@code keydown} and {@code keyup} in milliseconds. Defaults to 0.
     */
    public Double delay;
    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to {@code false}.
     */
    public Boolean noWaitAfter;
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass {@code 0} to disable timeout. The default value can be changed by
     * using the {@link BrowserContext#setDefaultTimeout BrowserContext.setDefaultTimeout()} or {@link Page#setDefaultTimeout
     * Page.setDefaultTimeout()} methods.
     */
    public Double timeout;

    public PressOptions setDelay(double delay) {
      this.delay = delay;
      return this;
    }
    public PressOptions setNoWaitAfter(boolean noWaitAfter) {
      this.noWaitAfter = noWaitAfter;
      return this;
    }
    public PressOptions setTimeout(double timeout) {
      this.timeout = timeout;
      return this;
    }
  }
  class ScreenshotOptions {
    /**
     * Hides default white background and allows capturing screenshots with transparency. Not applicable to {@code jpeg} images.
     * Defaults to {@code false}.
     */
    public Boolean omitBackground;
    /**
     * The file path to save the image to. The screenshot type will be inferred from file extension. If {@code path} is a relative
     * path, then it is resolved relative to the current working directory. If no path is provided, the image won't be saved to
     * the disk.
     */
    public Path path;
    /**
     * The quality of the image, between 0-100. Not applicable to {@code png} images.
     */
    public Integer quality;
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass {@code 0} to disable timeout. The default value can be changed by
     * using the {@link BrowserContext#setDefaultTimeout BrowserContext.setDefaultTimeout()} or {@link Page#setDefaultTimeout
     * Page.setDefaultTimeout()} methods.
     */
    public Double timeout;
    /**
     * Specify screenshot type, defaults to {@code png}.
     */
    public ScreenshotType type;

    public ScreenshotOptions setOmitBackground(boolean omitBackground) {
      this.omitBackground = omitBackground;
      return this;
    }
    public ScreenshotOptions setPath(Path path) {
      this.path = path;
      return this;
    }
    public ScreenshotOptions setQuality(int quality) {
      this.quality = quality;
      return this;
    }
    public ScreenshotOptions setTimeout(double timeout) {
      this.timeout = timeout;
      return this;
    }
    public ScreenshotOptions setType(ScreenshotType type) {
      this.type = type;
      return this;
    }
  }
  class ScrollIntoViewIfNeededOptions {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass {@code 0} to disable timeout. The default value can be changed by
     * using the {@link BrowserContext#setDefaultTimeout BrowserContext.setDefaultTimeout()} or {@link Page#setDefaultTimeout
     * Page.setDefaultTimeout()} methods.
     */
    public Double timeout;

    public ScrollIntoViewIfNeededOptions setTimeout(double timeout) {
      this.timeout = timeout;
      return this;
    }
  }
  class SelectOptionOptions {
    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to {@code false}.
     */
    public Boolean noWaitAfter;
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass {@code 0} to disable timeout. The default value can be changed by
     * using the {@link BrowserContext#setDefaultTimeout BrowserContext.setDefaultTimeout()} or {@link Page#setDefaultTimeout
     * Page.setDefaultTimeout()} methods.
     */
    public Double timeout;

    public SelectOptionOptions setNoWaitAfter(boolean noWaitAfter) {
      this.noWaitAfter = noWaitAfter;
      return this;
    }
    public SelectOptionOptions setTimeout(double timeout) {
      this.timeout = timeout;
      return this;
    }
  }
  class SelectTextOptions {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass {@code 0} to disable timeout. The default value can be changed by
     * using the {@link BrowserContext#setDefaultTimeout BrowserContext.setDefaultTimeout()} or {@link Page#setDefaultTimeout
     * Page.setDefaultTimeout()} methods.
     */
    public Double timeout;

    public SelectTextOptions setTimeout(double timeout) {
      this.timeout = timeout;
      return this;
    }
  }
  class SetInputFilesOptions {
    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to {@code false}.
     */
    public Boolean noWaitAfter;
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass {@code 0} to disable timeout. The default value can be changed by
     * using the {@link BrowserContext#setDefaultTimeout BrowserContext.setDefaultTimeout()} or {@link Page#setDefaultTimeout
     * Page.setDefaultTimeout()} methods.
     */
    public Double timeout;

    public SetInputFilesOptions setNoWaitAfter(boolean noWaitAfter) {
      this.noWaitAfter = noWaitAfter;
      return this;
    }
    public SetInputFilesOptions setTimeout(double timeout) {
      this.timeout = timeout;
      return this;
    }
  }
  class TapOptions {
    /**
     * Whether to bypass the <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks. Defaults to
     * {@code false}.
     */
    public Boolean force;
    /**
     * Modifier keys to press. Ensures that only these modifiers are pressed during the operation, and then restores current
     * modifiers back. If not specified, currently pressed modifiers are used.
     */
    public List<KeyboardModifier> modifiers;
    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to {@code false}.
     */
    public Boolean noWaitAfter;
    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of the
     * element.
     */
    public Position position;
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass {@code 0} to disable timeout. The default value can be changed by
     * using the {@link BrowserContext#setDefaultTimeout BrowserContext.setDefaultTimeout()} or {@link Page#setDefaultTimeout
     * Page.setDefaultTimeout()} methods.
     */
    public Double timeout;
    /**
     * When set, this method only performs the <a href="https://playwright.dev/java/docs/actionability/">actionability</a>
     * checks and skips the action. Defaults to {@code false}. Useful to wait until the element is ready for the action without
     * performing it.
     */
    public Boolean trial;

    public TapOptions setForce(boolean force) {
      this.force = force;
      return this;
    }
    public TapOptions setModifiers(List<KeyboardModifier> modifiers) {
      this.modifiers = modifiers;
      return this;
    }
    public TapOptions setNoWaitAfter(boolean noWaitAfter) {
      this.noWaitAfter = noWaitAfter;
      return this;
    }
    public TapOptions setPosition(double x, double y) {
      return setPosition(new Position(x, y));
    }
    public TapOptions setPosition(Position position) {
      this.position = position;
      return this;
    }
    public TapOptions setTimeout(double timeout) {
      this.timeout = timeout;
      return this;
    }
    public TapOptions setTrial(boolean trial) {
      this.trial = trial;
      return this;
    }
  }
  class TypeOptions {
    /**
     * Time to wait between key presses in milliseconds. Defaults to 0.
     */
    public Double delay;
    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to {@code false}.
     */
    public Boolean noWaitAfter;
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass {@code 0} to disable timeout. The default value can be changed by
     * using the {@link BrowserContext#setDefaultTimeout BrowserContext.setDefaultTimeout()} or {@link Page#setDefaultTimeout
     * Page.setDefaultTimeout()} methods.
     */
    public Double timeout;

    public TypeOptions setDelay(double delay) {
      this.delay = delay;
      return this;
    }
    public TypeOptions setNoWaitAfter(boolean noWaitAfter) {
      this.noWaitAfter = noWaitAfter;
      return this;
    }
    public TypeOptions setTimeout(double timeout) {
      this.timeout = timeout;
      return this;
    }
  }
  class UncheckOptions {
    /**
     * Whether to bypass the <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks. Defaults to
     * {@code false}.
     */
    public Boolean force;
    /**
     * Actions that initiate navigations are waiting for these navigations to happen and for pages to start loading. You can
     * opt out of waiting via setting this flag. You would only need this option in the exceptional cases such as navigating to
     * inaccessible pages. Defaults to {@code false}.
     */
    public Boolean noWaitAfter;
    /**
     * A point to use relative to the top-left corner of element padding box. If not specified, uses some visible point of the
     * element.
     */
    public Position position;
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass {@code 0} to disable timeout. The default value can be changed by
     * using the {@link BrowserContext#setDefaultTimeout BrowserContext.setDefaultTimeout()} or {@link Page#setDefaultTimeout
     * Page.setDefaultTimeout()} methods.
     */
    public Double timeout;
    /**
     * When set, this method only performs the <a href="https://playwright.dev/java/docs/actionability/">actionability</a>
     * checks and skips the action. Defaults to {@code false}. Useful to wait until the element is ready for the action without
     * performing it.
     */
    public Boolean trial;

    public UncheckOptions setForce(boolean force) {
      this.force = force;
      return this;
    }
    public UncheckOptions setNoWaitAfter(boolean noWaitAfter) {
      this.noWaitAfter = noWaitAfter;
      return this;
    }
    public UncheckOptions setPosition(double x, double y) {
      return setPosition(new Position(x, y));
    }
    public UncheckOptions setPosition(Position position) {
      this.position = position;
      return this;
    }
    public UncheckOptions setTimeout(double timeout) {
      this.timeout = timeout;
      return this;
    }
    public UncheckOptions setTrial(boolean trial) {
      this.trial = trial;
      return this;
    }
  }
  class WaitForElementStateOptions {
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass {@code 0} to disable timeout. The default value can be changed by
     * using the {@link BrowserContext#setDefaultTimeout BrowserContext.setDefaultTimeout()} or {@link Page#setDefaultTimeout
     * Page.setDefaultTimeout()} methods.
     */
    public Double timeout;

    public WaitForElementStateOptions setTimeout(double timeout) {
      this.timeout = timeout;
      return this;
    }
  }
  class WaitForSelectorOptions {
    /**
     * Defaults to {@code "visible"}. Can be either:
     * <ul>
     * <li> {@code "attached"} - wait for element to be present in DOM.</li>
     * <li> {@code "detached"} - wait for element to not be present in DOM.</li>
     * <li> {@code "visible"} - wait for element to have non-empty bounding box and no {@code visibility:hidden}. Note that element without any
     * content or with {@code display:none} has an empty bounding box and is not considered visible.</li>
     * <li> {@code "hidden"} - wait for element to be either detached from DOM, or have an empty bounding box or {@code visibility:hidden}. This
     * is opposite to the {@code "visible"} option.</li>
     * </ul>
     */
    public WaitForSelectorState state;
    /**
     * Maximum time in milliseconds, defaults to 30 seconds, pass {@code 0} to disable timeout. The default value can be changed by
     * using the {@link BrowserContext#setDefaultTimeout BrowserContext.setDefaultTimeout()} or {@link Page#setDefaultTimeout
     * Page.setDefaultTimeout()} methods.
     */
    public Double timeout;

    public WaitForSelectorOptions setState(WaitForSelectorState state) {
      this.state = state;
      return this;
    }
    public WaitForSelectorOptions setTimeout(double timeout) {
      this.timeout = timeout;
      return this;
    }
  }
  /**
   * This method returns the bounding box of the element, or {@code null} if the element is not visible. The bounding box is
   * calculated relative to the main frame viewport - which is usually the same as the browser window.
   *
   * <p> Scrolling affects the returned bonding box, similarly to <a
   * href="https://developer.mozilla.org/en-US/docs/Web/API/Element/getBoundingClientRect">Element.getBoundingClientRect</a>.
   * That means {@code x} and/or {@code y} may be negative.
   *
   * <p> Elements from child frames return the bounding box relative to the main frame, unlike the <a
   * href="https://developer.mozilla.org/en-US/docs/Web/API/Element/getBoundingClientRect">Element.getBoundingClientRect</a>.
   *
   * <p> Assuming the page is static, it is safe to use bounding box coordinates to perform input. For example, the following
   * snippet should click the center of the element.
   * <pre>{@code
   * BoundingBox box = elementHandle.boundingBox();
   * page.mouse().click(box.x + box.width / 2, box.y + box.height / 2);
   * }</pre>
   */
  BoundingBox boundingBox();
  /**
   * This method checks the element by performing the following steps:
   * <ol>
   * <li> Ensure that element is a checkbox or a radio input. If not, this method throws. If the element is already checked, this
   * method returns immediately.</li>
   * <li> Wait for <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks on the element, unless
   * {@code force} option is set.</li>
   * <li> Scroll the element into view if needed.</li>
   * <li> Use {@link Page#mouse Page.mouse()} to click in the center of the element.</li>
   * <li> Wait for initiated navigations to either succeed or fail, unless {@code noWaitAfter} option is set.</li>
   * <li> Ensure that the element is now checked. If not, this method throws.</li>
   * </ol>
   *
   * <p> If the element is detached from the DOM at any moment during the action, this method throws.
   *
   * <p> When all steps combined have not finished during the specified {@code timeout}, this method throws a {@code TimeoutError}. Passing
   * zero timeout disables this.
   */
  default void check() {
    check(null);
  }
  /**
   * This method checks the element by performing the following steps:
   * <ol>
   * <li> Ensure that element is a checkbox or a radio input. If not, this method throws. If the element is already checked, this
   * method returns immediately.</li>
   * <li> Wait for <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks on the element, unless
   * {@code force} option is set.</li>
   * <li> Scroll the element into view if needed.</li>
   * <li> Use {@link Page#mouse Page.mouse()} to click in the center of the element.</li>
   * <li> Wait for initiated navigations to either succeed or fail, unless {@code noWaitAfter} option is set.</li>
   * <li> Ensure that the element is now checked. If not, this method throws.</li>
   * </ol>
   *
   * <p> If the element is detached from the DOM at any moment during the action, this method throws.
   *
   * <p> When all steps combined have not finished during the specified {@code timeout}, this method throws a {@code TimeoutError}. Passing
   * zero timeout disables this.
   */
  void check(CheckOptions options);
  /**
   * This method clicks the element by performing the following steps:
   * <ol>
   * <li> Wait for <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks on the element, unless
   * {@code force} option is set.</li>
   * <li> Scroll the element into view if needed.</li>
   * <li> Use {@link Page#mouse Page.mouse()} to click in the center of the element, or the specified {@code position}.</li>
   * <li> Wait for initiated navigations to either succeed or fail, unless {@code noWaitAfter} option is set.</li>
   * </ol>
   *
   * <p> If the element is detached from the DOM at any moment during the action, this method throws.
   *
   * <p> When all steps combined have not finished during the specified {@code timeout}, this method throws a {@code TimeoutError}. Passing
   * zero timeout disables this.
   */
  default void click() {
    click(null);
  }
  /**
   * This method clicks the element by performing the following steps:
   * <ol>
   * <li> Wait for <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks on the element, unless
   * {@code force} option is set.</li>
   * <li> Scroll the element into view if needed.</li>
   * <li> Use {@link Page#mouse Page.mouse()} to click in the center of the element, or the specified {@code position}.</li>
   * <li> Wait for initiated navigations to either succeed or fail, unless {@code noWaitAfter} option is set.</li>
   * </ol>
   *
   * <p> If the element is detached from the DOM at any moment during the action, this method throws.
   *
   * <p> When all steps combined have not finished during the specified {@code timeout}, this method throws a {@code TimeoutError}. Passing
   * zero timeout disables this.
   */
  void click(ClickOptions options);
  /**
   * Returns the content frame for element handles referencing iframe nodes, or {@code null} otherwise
   */
  Frame contentFrame();
  /**
   * This method double clicks the element by performing the following steps:
   * <ol>
   * <li> Wait for <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks on the element, unless
   * {@code force} option is set.</li>
   * <li> Scroll the element into view if needed.</li>
   * <li> Use {@link Page#mouse Page.mouse()} to double click in the center of the element, or the specified {@code position}.</li>
   * <li> Wait for initiated navigations to either succeed or fail, unless {@code noWaitAfter} option is set. Note that if the first
   * click of the {@code dblclick()} triggers a navigation event, this method will throw.</li>
   * </ol>
   *
   * <p> If the element is detached from the DOM at any moment during the action, this method throws.
   *
   * <p> When all steps combined have not finished during the specified {@code timeout}, this method throws a {@code TimeoutError}. Passing
   * zero timeout disables this.
   *
   * <p> <strong>NOTE:</strong> {@code elementHandle.dblclick()} dispatches two {@code click} events and a single {@code dblclick} event.
   */
  default void dblclick() {
    dblclick(null);
  }
  /**
   * This method double clicks the element by performing the following steps:
   * <ol>
   * <li> Wait for <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks on the element, unless
   * {@code force} option is set.</li>
   * <li> Scroll the element into view if needed.</li>
   * <li> Use {@link Page#mouse Page.mouse()} to double click in the center of the element, or the specified {@code position}.</li>
   * <li> Wait for initiated navigations to either succeed or fail, unless {@code noWaitAfter} option is set. Note that if the first
   * click of the {@code dblclick()} triggers a navigation event, this method will throw.</li>
   * </ol>
   *
   * <p> If the element is detached from the DOM at any moment during the action, this method throws.
   *
   * <p> When all steps combined have not finished during the specified {@code timeout}, this method throws a {@code TimeoutError}. Passing
   * zero timeout disables this.
   *
   * <p> <strong>NOTE:</strong> {@code elementHandle.dblclick()} dispatches two {@code click} events and a single {@code dblclick} event.
   */
  void dblclick(DblclickOptions options);
  /**
   * The snippet below dispatches the {@code click} event on the element. Regardless of the visibility state of the element,
   * {@code click} is dispatched. This is equivalent to calling <a
   * href="https://developer.mozilla.org/en-US/docs/Web/API/HTMLElement/click">element.click()</a>.
   * <pre>{@code
   * elementHandle.dispatchEvent("click");
   * }</pre>
   *
   * <p> Under the hood, it creates an instance of an event based on the given {@code type}, initializes it with {@code eventInit} properties
   * and dispatches it on the element. Events are {@code composed}, {@code cancelable} and bubble by default.
   *
   * <p> Since {@code eventInit} is event-specific, please refer to the events documentation for the lists of initial properties:
   * <ul>
   * <li> <a href="https://developer.mozilla.org/en-US/docs/Web/API/DragEvent/DragEvent">DragEvent</a></li>
   * <li> <a href="https://developer.mozilla.org/en-US/docs/Web/API/FocusEvent/FocusEvent">FocusEvent</a></li>
   * <li> <a href="https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/KeyboardEvent">KeyboardEvent</a></li>
   * <li> <a href="https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent/MouseEvent">MouseEvent</a></li>
   * <li> <a href="https://developer.mozilla.org/en-US/docs/Web/API/PointerEvent/PointerEvent">PointerEvent</a></li>
   * <li> <a href="https://developer.mozilla.org/en-US/docs/Web/API/TouchEvent/TouchEvent">TouchEvent</a></li>
   * <li> <a href="https://developer.mozilla.org/en-US/docs/Web/API/Event/Event">Event</a></li>
   * </ul>
   *
   * <p> You can also specify {@code JSHandle} as the property value if you want live objects to be passed into the event:
   * <pre>{@code
   * // Note you can only create DataTransfer in Chromium and Firefox
   * JSHandle dataTransfer = page.evaluateHandle("() => new DataTransfer()");
   * Map<String, Object> arg = new HashMap<>();
   * arg.put("dataTransfer", dataTransfer);
   * elementHandle.dispatchEvent("dragstart", arg);
   * }</pre>
   *
   * @param type DOM event type: {@code "click"}, {@code "dragstart"}, etc.
   */
  default void dispatchEvent(String type) {
    dispatchEvent(type, null);
  }
  /**
   * The snippet below dispatches the {@code click} event on the element. Regardless of the visibility state of the element,
   * {@code click} is dispatched. This is equivalent to calling <a
   * href="https://developer.mozilla.org/en-US/docs/Web/API/HTMLElement/click">element.click()</a>.
   * <pre>{@code
   * elementHandle.dispatchEvent("click");
   * }</pre>
   *
   * <p> Under the hood, it creates an instance of an event based on the given {@code type}, initializes it with {@code eventInit} properties
   * and dispatches it on the element. Events are {@code composed}, {@code cancelable} and bubble by default.
   *
   * <p> Since {@code eventInit} is event-specific, please refer to the events documentation for the lists of initial properties:
   * <ul>
   * <li> <a href="https://developer.mozilla.org/en-US/docs/Web/API/DragEvent/DragEvent">DragEvent</a></li>
   * <li> <a href="https://developer.mozilla.org/en-US/docs/Web/API/FocusEvent/FocusEvent">FocusEvent</a></li>
   * <li> <a href="https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/KeyboardEvent">KeyboardEvent</a></li>
   * <li> <a href="https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent/MouseEvent">MouseEvent</a></li>
   * <li> <a href="https://developer.mozilla.org/en-US/docs/Web/API/PointerEvent/PointerEvent">PointerEvent</a></li>
   * <li> <a href="https://developer.mozilla.org/en-US/docs/Web/API/TouchEvent/TouchEvent">TouchEvent</a></li>
   * <li> <a href="https://developer.mozilla.org/en-US/docs/Web/API/Event/Event">Event</a></li>
   * </ul>
   *
   * <p> You can also specify {@code JSHandle} as the property value if you want live objects to be passed into the event:
   * <pre>{@code
   * // Note you can only create DataTransfer in Chromium and Firefox
   * JSHandle dataTransfer = page.evaluateHandle("() => new DataTransfer()");
   * Map<String, Object> arg = new HashMap<>();
   * arg.put("dataTransfer", dataTransfer);
   * elementHandle.dispatchEvent("dragstart", arg);
   * }</pre>
   *
   * @param type DOM event type: {@code "click"}, {@code "dragstart"}, etc.
   * @param eventInit Optional event-specific initialization properties.
   */
  void dispatchEvent(String type, Object eventInit);
  /**
   * Returns the return value of {@code expression}.
   *
   * <p> The method finds an element matching the specified selector in the {@code ElementHandle}s subtree and passes it as a first
   * argument to {@code expression}. See <a href="https://playwright.dev/java/docs/selectors/">Working with selectors</a> for more
   * details. If no elements match the selector, the method throws an error.
   *
   * <p> If {@code expression} returns a <a
   * href='https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Promise'>Promise</a>, then {@link
   * ElementHandle#evalOnSelector ElementHandle.evalOnSelector()} would wait for the promise to resolve and return its value.
   *
   * <p> Examples:
   * <pre>{@code
   * ElementHandle tweetHandle = page.querySelector(".tweet");
   * assertEquals("100", tweetHandle.evalOnSelector(".like", "node => node.innerText"));
   * assertEquals("10", tweetHandle.evalOnSelector(".retweets", "node => node.innerText"));
   * }</pre>
   *
   * @param selector A selector to query for. See <a href="https://playwright.dev/java/docs/selectors/">working with selectors</a> for more
   * details.
   * @param expression JavaScript expression to be evaluated in the browser context. If it looks like a function declaration, it is interpreted
   * as a function. Otherwise, evaluated as an expression.
   */
  default Object evalOnSelector(String selector, String expression) {
    return evalOnSelector(selector, expression, null);
  }
  /**
   * Returns the return value of {@code expression}.
   *
   * <p> The method finds an element matching the specified selector in the {@code ElementHandle}s subtree and passes it as a first
   * argument to {@code expression}. See <a href="https://playwright.dev/java/docs/selectors/">Working with selectors</a> for more
   * details. If no elements match the selector, the method throws an error.
   *
   * <p> If {@code expression} returns a <a
   * href='https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Promise'>Promise</a>, then {@link
   * ElementHandle#evalOnSelector ElementHandle.evalOnSelector()} would wait for the promise to resolve and return its value.
   *
   * <p> Examples:
   * <pre>{@code
   * ElementHandle tweetHandle = page.querySelector(".tweet");
   * assertEquals("100", tweetHandle.evalOnSelector(".like", "node => node.innerText"));
   * assertEquals("10", tweetHandle.evalOnSelector(".retweets", "node => node.innerText"));
   * }</pre>
   *
   * @param selector A selector to query for. See <a href="https://playwright.dev/java/docs/selectors/">working with selectors</a> for more
   * details.
   * @param expression JavaScript expression to be evaluated in the browser context. If it looks like a function declaration, it is interpreted
   * as a function. Otherwise, evaluated as an expression.
   * @param arg Optional argument to pass to {@code expression}.
   */
  Object evalOnSelector(String selector, String expression, Object arg);
  /**
   * Returns the return value of {@code expression}.
   *
   * <p> The method finds all elements matching the specified selector in the {@code ElementHandle}'s subtree and passes an array of
   * matched elements as a first argument to {@code expression}. See <a href="https://playwright.dev/java/docs/selectors/">Working
   * with selectors</a> for more details.
   *
   * <p> If {@code expression} returns a <a
   * href='https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Promise'>Promise</a>, then {@link
   * ElementHandle#evalOnSelectorAll ElementHandle.evalOnSelectorAll()} would wait for the promise to resolve and return its
   * value.
   *
   * <p> Examples:
   * <pre>{@code
   * ElementHandle feedHandle = page.querySelector(".feed");
   * assertEquals(Arrays.asList("Hello!", "Hi!"), feedHandle.evalOnSelectorAll(".tweet", "nodes => nodes.map(n => n.innerText)"));
   * }</pre>
   *
   * @param selector A selector to query for. See <a href="https://playwright.dev/java/docs/selectors/">working with selectors</a> for more
   * details.
   * @param expression JavaScript expression to be evaluated in the browser context. If it looks like a function declaration, it is interpreted
   * as a function. Otherwise, evaluated as an expression.
   */
  default Object evalOnSelectorAll(String selector, String expression) {
    return evalOnSelectorAll(selector, expression, null);
  }
  /**
   * Returns the return value of {@code expression}.
   *
   * <p> The method finds all elements matching the specified selector in the {@code ElementHandle}'s subtree and passes an array of
   * matched elements as a first argument to {@code expression}. See <a href="https://playwright.dev/java/docs/selectors/">Working
   * with selectors</a> for more details.
   *
   * <p> If {@code expression} returns a <a
   * href='https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Promise'>Promise</a>, then {@link
   * ElementHandle#evalOnSelectorAll ElementHandle.evalOnSelectorAll()} would wait for the promise to resolve and return its
   * value.
   *
   * <p> Examples:
   * <pre>{@code
   * ElementHandle feedHandle = page.querySelector(".feed");
   * assertEquals(Arrays.asList("Hello!", "Hi!"), feedHandle.evalOnSelectorAll(".tweet", "nodes => nodes.map(n => n.innerText)"));
   * }</pre>
   *
   * @param selector A selector to query for. See <a href="https://playwright.dev/java/docs/selectors/">working with selectors</a> for more
   * details.
   * @param expression JavaScript expression to be evaluated in the browser context. If it looks like a function declaration, it is interpreted
   * as a function. Otherwise, evaluated as an expression.
   * @param arg Optional argument to pass to {@code expression}.
   */
  Object evalOnSelectorAll(String selector, String expression, Object arg);
  /**
   * This method waits for <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks, focuses the
   * element, fills it and triggers an {@code input} event after filling. Note that you can pass an empty string to clear the input
   * field.
   *
   * <p> If the target element is not an {@code <input>}, {@code <textarea>} or {@code [contenteditable]} element, this method throws an error.
   * However, if the element is inside the {@code <label>} element that has an associated <a
   * href="https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control">control</a>, the control will be filled
   * instead.
   *
   * <p> To send fine-grained keyboard events, use {@link ElementHandle#type ElementHandle.type()}.
   *
   * @param value Value to set for the {@code <input>}, {@code <textarea>} or {@code [contenteditable]} element.
   */
  default void fill(String value) {
    fill(value, null);
  }
  /**
   * This method waits for <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks, focuses the
   * element, fills it and triggers an {@code input} event after filling. Note that you can pass an empty string to clear the input
   * field.
   *
   * <p> If the target element is not an {@code <input>}, {@code <textarea>} or {@code [contenteditable]} element, this method throws an error.
   * However, if the element is inside the {@code <label>} element that has an associated <a
   * href="https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control">control</a>, the control will be filled
   * instead.
   *
   * <p> To send fine-grained keyboard events, use {@link ElementHandle#type ElementHandle.type()}.
   *
   * @param value Value to set for the {@code <input>}, {@code <textarea>} or {@code [contenteditable]} element.
   */
  void fill(String value, FillOptions options);
  /**
   * Calls <a href="https://developer.mozilla.org/en-US/docs/Web/API/HTMLElement/focus">focus</a> on the element.
   */
  void focus();
  /**
   * Returns element attribute value.
   *
   * @param name Attribute name to get the value for.
   */
  String getAttribute(String name);
  /**
   * This method hovers over the element by performing the following steps:
   * <ol>
   * <li> Wait for <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks on the element, unless
   * {@code force} option is set.</li>
   * <li> Scroll the element into view if needed.</li>
   * <li> Use {@link Page#mouse Page.mouse()} to hover over the center of the element, or the specified {@code position}.</li>
   * <li> Wait for initiated navigations to either succeed or fail, unless {@code noWaitAfter} option is set.</li>
   * </ol>
   *
   * <p> If the element is detached from the DOM at any moment during the action, this method throws.
   *
   * <p> When all steps combined have not finished during the specified {@code timeout}, this method throws a {@code TimeoutError}. Passing
   * zero timeout disables this.
   */
  default void hover() {
    hover(null);
  }
  /**
   * This method hovers over the element by performing the following steps:
   * <ol>
   * <li> Wait for <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks on the element, unless
   * {@code force} option is set.</li>
   * <li> Scroll the element into view if needed.</li>
   * <li> Use {@link Page#mouse Page.mouse()} to hover over the center of the element, or the specified {@code position}.</li>
   * <li> Wait for initiated navigations to either succeed or fail, unless {@code noWaitAfter} option is set.</li>
   * </ol>
   *
   * <p> If the element is detached from the DOM at any moment during the action, this method throws.
   *
   * <p> When all steps combined have not finished during the specified {@code timeout}, this method throws a {@code TimeoutError}. Passing
   * zero timeout disables this.
   */
  void hover(HoverOptions options);
  /**
   * Returns the {@code element.innerHTML}.
   */
  String innerHTML();
  /**
   * Returns the {@code element.innerText}.
   */
  String innerText();
  /**
   * Returns whether the element is checked. Throws if the element is not a checkbox or radio input.
   */
  boolean isChecked();
  /**
   * Returns whether the element is disabled, the opposite of <a
   * href="https://playwright.dev/java/docs/actionability/#enabled">enabled</a>.
   */
  boolean isDisabled();
  /**
   * Returns whether the element is <a href="https://playwright.dev/java/docs/actionability/#editable">editable</a>.
   */
  boolean isEditable();
  /**
   * Returns whether the element is <a href="https://playwright.dev/java/docs/actionability/#enabled">enabled</a>.
   */
  boolean isEnabled();
  /**
   * Returns whether the element is hidden, the opposite of <a
   * href="https://playwright.dev/java/docs/actionability/#visible">visible</a>.
   */
  boolean isHidden();
  /**
   * Returns whether the element is <a href="https://playwright.dev/java/docs/actionability/#visible">visible</a>.
   */
  boolean isVisible();
  /**
   * Returns the frame containing the given element.
   */
  Frame ownerFrame();
  /**
   * Focuses the element, and then uses {@link Keyboard#down Keyboard.down()} and {@link Keyboard#up Keyboard.up()}.
   *
   * <p> {@code key} can specify the intended <a
   * href="https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/key">keyboardEvent.key</a> value or a single
   * character to generate the text for. A superset of the {@code key} values can be found <a
   * href="https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/key/Key_Values">here</a>. Examples of the keys are:
   *
   * <p> {@code F1} - {@code F12}, {@code Digit0}- {@code Digit9}, {@code KeyA}- {@code KeyZ}, {@code Backquote}, {@code Minus}, {@code Equal}, {@code Backslash}, {@code Backspace}, {@code Tab},
   * {@code Delete}, {@code Escape}, {@code ArrowDown}, {@code End}, {@code Enter}, {@code Home}, {@code Insert}, {@code PageDown}, {@code PageUp}, {@code ArrowRight}, {@code ArrowUp}, etc.
   *
   * <p> Following modification shortcuts are also supported: {@code Shift}, {@code Control}, {@code Alt}, {@code Meta}, {@code ShiftLeft}.
   *
   * <p> Holding down {@code Shift} will type the text that corresponds to the {@code key} in the upper case.
   *
   * <p> If {@code key} is a single character, it is case-sensitive, so the values {@code a} and {@code A} will generate different respective
   * texts.
   *
   * <p> Shortcuts such as {@code key: "Control+o"} or {@code key: "Control+Shift+T"} are supported as well. When specified with the
   * modifier, modifier is pressed and being held while the subsequent key is being pressed.
   *
   * @param key Name of the key to press or a character to generate, such as {@code ArrowLeft} or {@code a}.
   */
  default void press(String key) {
    press(key, null);
  }
  /**
   * Focuses the element, and then uses {@link Keyboard#down Keyboard.down()} and {@link Keyboard#up Keyboard.up()}.
   *
   * <p> {@code key} can specify the intended <a
   * href="https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/key">keyboardEvent.key</a> value or a single
   * character to generate the text for. A superset of the {@code key} values can be found <a
   * href="https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/key/Key_Values">here</a>. Examples of the keys are:
   *
   * <p> {@code F1} - {@code F12}, {@code Digit0}- {@code Digit9}, {@code KeyA}- {@code KeyZ}, {@code Backquote}, {@code Minus}, {@code Equal}, {@code Backslash}, {@code Backspace}, {@code Tab},
   * {@code Delete}, {@code Escape}, {@code ArrowDown}, {@code End}, {@code Enter}, {@code Home}, {@code Insert}, {@code PageDown}, {@code PageUp}, {@code ArrowRight}, {@code ArrowUp}, etc.
   *
   * <p> Following modification shortcuts are also supported: {@code Shift}, {@code Control}, {@code Alt}, {@code Meta}, {@code ShiftLeft}.
   *
   * <p> Holding down {@code Shift} will type the text that corresponds to the {@code key} in the upper case.
   *
   * <p> If {@code key} is a single character, it is case-sensitive, so the values {@code a} and {@code A} will generate different respective
   * texts.
   *
   * <p> Shortcuts such as {@code key: "Control+o"} or {@code key: "Control+Shift+T"} are supported as well. When specified with the
   * modifier, modifier is pressed and being held while the subsequent key is being pressed.
   *
   * @param key Name of the key to press or a character to generate, such as {@code ArrowLeft} or {@code a}.
   */
  void press(String key, PressOptions options);
  /**
   * The method finds an element matching the specified selector in the {@code ElementHandle}'s subtree. See <a
   * href="https://playwright.dev/java/docs/selectors/">Working with selectors</a> for more details. If no elements match the
   * selector, returns {@code null}.
   *
   * @param selector A selector to query for. See <a href="https://playwright.dev/java/docs/selectors/">working with selectors</a> for more
   * details.
   */
  ElementHandle querySelector(String selector);
  /**
   * The method finds all elements matching the specified selector in the {@code ElementHandle}s subtree. See <a
   * href="https://playwright.dev/java/docs/selectors/">Working with selectors</a> for more details. If no elements match the
   * selector, returns empty array.
   *
   * @param selector A selector to query for. See <a href="https://playwright.dev/java/docs/selectors/">working with selectors</a> for more
   * details.
   */
  List<ElementHandle> querySelectorAll(String selector);
  /**
   * Returns the buffer with the captured screenshot.
   *
   * <p> This method waits for the <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks, then
   * scrolls element into view before taking a screenshot. If the element is detached from DOM, the method throws an error.
   */
  default byte[] screenshot() {
    return screenshot(null);
  }
  /**
   * Returns the buffer with the captured screenshot.
   *
   * <p> This method waits for the <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks, then
   * scrolls element into view before taking a screenshot. If the element is detached from DOM, the method throws an error.
   */
  byte[] screenshot(ScreenshotOptions options);
  /**
   * This method waits for <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks, then tries to
   * scroll element into view, unless it is completely visible as defined by <a
   * href="https://developer.mozilla.org/en-US/docs/Web/API/Intersection_Observer_API">IntersectionObserver</a>'s {@code ratio}.
   *
   * <p> Throws when {@code elementHandle} does not point to an element <a
   * href="https://developer.mozilla.org/en-US/docs/Web/API/Node/isConnected">connected</a> to a Document or a ShadowRoot.
   */
  default void scrollIntoViewIfNeeded() {
    scrollIntoViewIfNeeded(null);
  }
  /**
   * This method waits for <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks, then tries to
   * scroll element into view, unless it is completely visible as defined by <a
   * href="https://developer.mozilla.org/en-US/docs/Web/API/Intersection_Observer_API">IntersectionObserver</a>'s {@code ratio}.
   *
   * <p> Throws when {@code elementHandle} does not point to an element <a
   * href="https://developer.mozilla.org/en-US/docs/Web/API/Node/isConnected">connected</a> to a Document or a ShadowRoot.
   */
  void scrollIntoViewIfNeeded(ScrollIntoViewIfNeededOptions options);
  /**
   * This method waits for <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks, waits until
   * all specified options are present in the {@code <select>} element and selects these options.
   *
   * <p> If the target element is not a {@code <select>} element, this method throws an error. However, if the element is inside the
   * {@code <label>} element that has an associated <a
   * href="https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control">control</a>, the control will be used
   * instead.
   *
   * <p> Returns the array of option values that have been successfully selected.
   *
   * <p> Triggers a {@code change} and {@code input} event once all the provided options have been selected.
   * <pre>{@code
   * // single selection matching the value
   * handle.selectOption("blue");
   * // single selection matching the label
   * handle.selectOption(new SelectOption().setLabel("Blue"));
   * // multiple selection
   * handle.selectOption(new String[] {"red", "green", "blue"});
   * }</pre>
   *
   * @param values Options to select. If the {@code <select>} has the {@code multiple} attribute, all matching options are selected, otherwise only the
   * first option matching one of the passed options is selected. String values are equivalent to {@code {value:'string'}}. Option
   * is considered matching if all specified properties match.
   */
  default List<String> selectOption(String values) {
    return selectOption(values, null);
  }
  /**
   * This method waits for <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks, waits until
   * all specified options are present in the {@code <select>} element and selects these options.
   *
   * <p> If the target element is not a {@code <select>} element, this method throws an error. However, if the element is inside the
   * {@code <label>} element that has an associated <a
   * href="https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control">control</a>, the control will be used
   * instead.
   *
   * <p> Returns the array of option values that have been successfully selected.
   *
   * <p> Triggers a {@code change} and {@code input} event once all the provided options have been selected.
   * <pre>{@code
   * // single selection matching the value
   * handle.selectOption("blue");
   * // single selection matching the label
   * handle.selectOption(new SelectOption().setLabel("Blue"));
   * // multiple selection
   * handle.selectOption(new String[] {"red", "green", "blue"});
   * }</pre>
   *
   * @param values Options to select. If the {@code <select>} has the {@code multiple} attribute, all matching options are selected, otherwise only the
   * first option matching one of the passed options is selected. String values are equivalent to {@code {value:'string'}}. Option
   * is considered matching if all specified properties match.
   */
  List<String> selectOption(String values, SelectOptionOptions options);
  /**
   * This method waits for <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks, waits until
   * all specified options are present in the {@code <select>} element and selects these options.
   *
   * <p> If the target element is not a {@code <select>} element, this method throws an error. However, if the element is inside the
   * {@code <label>} element that has an associated <a
   * href="https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control">control</a>, the control will be used
   * instead.
   *
   * <p> Returns the array of option values that have been successfully selected.
   *
   * <p> Triggers a {@code change} and {@code input} event once all the provided options have been selected.
   * <pre>{@code
   * // single selection matching the value
   * handle.selectOption("blue");
   * // single selection matching the label
   * handle.selectOption(new SelectOption().setLabel("Blue"));
   * // multiple selection
   * handle.selectOption(new String[] {"red", "green", "blue"});
   * }</pre>
   *
   * @param values Options to select. If the {@code <select>} has the {@code multiple} attribute, all matching options are selected, otherwise only the
   * first option matching one of the passed options is selected. String values are equivalent to {@code {value:'string'}}. Option
   * is considered matching if all specified properties match.
   */
  default List<String> selectOption(ElementHandle values) {
    return selectOption(values, null);
  }
  /**
   * This method waits for <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks, waits until
   * all specified options are present in the {@code <select>} element and selects these options.
   *
   * <p> If the target element is not a {@code <select>} element, this method throws an error. However, if the element is inside the
   * {@code <label>} element that has an associated <a
   * href="https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control">control</a>, the control will be used
   * instead.
   *
   * <p> Returns the array of option values that have been successfully selected.
   *
   * <p> Triggers a {@code change} and {@code input} event once all the provided options have been selected.
   * <pre>{@code
   * // single selection matching the value
   * handle.selectOption("blue");
   * // single selection matching the label
   * handle.selectOption(new SelectOption().setLabel("Blue"));
   * // multiple selection
   * handle.selectOption(new String[] {"red", "green", "blue"});
   * }</pre>
   *
   * @param values Options to select. If the {@code <select>} has the {@code multiple} attribute, all matching options are selected, otherwise only the
   * first option matching one of the passed options is selected. String values are equivalent to {@code {value:'string'}}. Option
   * is considered matching if all specified properties match.
   */
  List<String> selectOption(ElementHandle values, SelectOptionOptions options);
  /**
   * This method waits for <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks, waits until
   * all specified options are present in the {@code <select>} element and selects these options.
   *
   * <p> If the target element is not a {@code <select>} element, this method throws an error. However, if the element is inside the
   * {@code <label>} element that has an associated <a
   * href="https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control">control</a>, the control will be used
   * instead.
   *
   * <p> Returns the array of option values that have been successfully selected.
   *
   * <p> Triggers a {@code change} and {@code input} event once all the provided options have been selected.
   * <pre>{@code
   * // single selection matching the value
   * handle.selectOption("blue");
   * // single selection matching the label
   * handle.selectOption(new SelectOption().setLabel("Blue"));
   * // multiple selection
   * handle.selectOption(new String[] {"red", "green", "blue"});
   * }</pre>
   *
   * @param values Options to select. If the {@code <select>} has the {@code multiple} attribute, all matching options are selected, otherwise only the
   * first option matching one of the passed options is selected. String values are equivalent to {@code {value:'string'}}. Option
   * is considered matching if all specified properties match.
   */
  default List<String> selectOption(String[] values) {
    return selectOption(values, null);
  }
  /**
   * This method waits for <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks, waits until
   * all specified options are present in the {@code <select>} element and selects these options.
   *
   * <p> If the target element is not a {@code <select>} element, this method throws an error. However, if the element is inside the
   * {@code <label>} element that has an associated <a
   * href="https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control">control</a>, the control will be used
   * instead.
   *
   * <p> Returns the array of option values that have been successfully selected.
   *
   * <p> Triggers a {@code change} and {@code input} event once all the provided options have been selected.
   * <pre>{@code
   * // single selection matching the value
   * handle.selectOption("blue");
   * // single selection matching the label
   * handle.selectOption(new SelectOption().setLabel("Blue"));
   * // multiple selection
   * handle.selectOption(new String[] {"red", "green", "blue"});
   * }</pre>
   *
   * @param values Options to select. If the {@code <select>} has the {@code multiple} attribute, all matching options are selected, otherwise only the
   * first option matching one of the passed options is selected. String values are equivalent to {@code {value:'string'}}. Option
   * is considered matching if all specified properties match.
   */
  List<String> selectOption(String[] values, SelectOptionOptions options);
  /**
   * This method waits for <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks, waits until
   * all specified options are present in the {@code <select>} element and selects these options.
   *
   * <p> If the target element is not a {@code <select>} element, this method throws an error. However, if the element is inside the
   * {@code <label>} element that has an associated <a
   * href="https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control">control</a>, the control will be used
   * instead.
   *
   * <p> Returns the array of option values that have been successfully selected.
   *
   * <p> Triggers a {@code change} and {@code input} event once all the provided options have been selected.
   * <pre>{@code
   * // single selection matching the value
   * handle.selectOption("blue");
   * // single selection matching the label
   * handle.selectOption(new SelectOption().setLabel("Blue"));
   * // multiple selection
   * handle.selectOption(new String[] {"red", "green", "blue"});
   * }</pre>
   *
   * @param values Options to select. If the {@code <select>} has the {@code multiple} attribute, all matching options are selected, otherwise only the
   * first option matching one of the passed options is selected. String values are equivalent to {@code {value:'string'}}. Option
   * is considered matching if all specified properties match.
   */
  default List<String> selectOption(SelectOption values) {
    return selectOption(values, null);
  }
  /**
   * This method waits for <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks, waits until
   * all specified options are present in the {@code <select>} element and selects these options.
   *
   * <p> If the target element is not a {@code <select>} element, this method throws an error. However, if the element is inside the
   * {@code <label>} element that has an associated <a
   * href="https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control">control</a>, the control will be used
   * instead.
   *
   * <p> Returns the array of option values that have been successfully selected.
   *
   * <p> Triggers a {@code change} and {@code input} event once all the provided options have been selected.
   * <pre>{@code
   * // single selection matching the value
   * handle.selectOption("blue");
   * // single selection matching the label
   * handle.selectOption(new SelectOption().setLabel("Blue"));
   * // multiple selection
   * handle.selectOption(new String[] {"red", "green", "blue"});
   * }</pre>
   *
   * @param values Options to select. If the {@code <select>} has the {@code multiple} attribute, all matching options are selected, otherwise only the
   * first option matching one of the passed options is selected. String values are equivalent to {@code {value:'string'}}. Option
   * is considered matching if all specified properties match.
   */
  List<String> selectOption(SelectOption values, SelectOptionOptions options);
  /**
   * This method waits for <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks, waits until
   * all specified options are present in the {@code <select>} element and selects these options.
   *
   * <p> If the target element is not a {@code <select>} element, this method throws an error. However, if the element is inside the
   * {@code <label>} element that has an associated <a
   * href="https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control">control</a>, the control will be used
   * instead.
   *
   * <p> Returns the array of option values that have been successfully selected.
   *
   * <p> Triggers a {@code change} and {@code input} event once all the provided options have been selected.
   * <pre>{@code
   * // single selection matching the value
   * handle.selectOption("blue");
   * // single selection matching the label
   * handle.selectOption(new SelectOption().setLabel("Blue"));
   * // multiple selection
   * handle.selectOption(new String[] {"red", "green", "blue"});
   * }</pre>
   *
   * @param values Options to select. If the {@code <select>} has the {@code multiple} attribute, all matching options are selected, otherwise only the
   * first option matching one of the passed options is selected. String values are equivalent to {@code {value:'string'}}. Option
   * is considered matching if all specified properties match.
   */
  default List<String> selectOption(ElementHandle[] values) {
    return selectOption(values, null);
  }
  /**
   * This method waits for <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks, waits until
   * all specified options are present in the {@code <select>} element and selects these options.
   *
   * <p> If the target element is not a {@code <select>} element, this method throws an error. However, if the element is inside the
   * {@code <label>} element that has an associated <a
   * href="https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control">control</a>, the control will be used
   * instead.
   *
   * <p> Returns the array of option values that have been successfully selected.
   *
   * <p> Triggers a {@code change} and {@code input} event once all the provided options have been selected.
   * <pre>{@code
   * // single selection matching the value
   * handle.selectOption("blue");
   * // single selection matching the label
   * handle.selectOption(new SelectOption().setLabel("Blue"));
   * // multiple selection
   * handle.selectOption(new String[] {"red", "green", "blue"});
   * }</pre>
   *
   * @param values Options to select. If the {@code <select>} has the {@code multiple} attribute, all matching options are selected, otherwise only the
   * first option matching one of the passed options is selected. String values are equivalent to {@code {value:'string'}}. Option
   * is considered matching if all specified properties match.
   */
  List<String> selectOption(ElementHandle[] values, SelectOptionOptions options);
  /**
   * This method waits for <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks, waits until
   * all specified options are present in the {@code <select>} element and selects these options.
   *
   * <p> If the target element is not a {@code <select>} element, this method throws an error. However, if the element is inside the
   * {@code <label>} element that has an associated <a
   * href="https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control">control</a>, the control will be used
   * instead.
   *
   * <p> Returns the array of option values that have been successfully selected.
   *
   * <p> Triggers a {@code change} and {@code input} event once all the provided options have been selected.
   * <pre>{@code
   * // single selection matching the value
   * handle.selectOption("blue");
   * // single selection matching the label
   * handle.selectOption(new SelectOption().setLabel("Blue"));
   * // multiple selection
   * handle.selectOption(new String[] {"red", "green", "blue"});
   * }</pre>
   *
   * @param values Options to select. If the {@code <select>} has the {@code multiple} attribute, all matching options are selected, otherwise only the
   * first option matching one of the passed options is selected. String values are equivalent to {@code {value:'string'}}. Option
   * is considered matching if all specified properties match.
   */
  default List<String> selectOption(SelectOption[] values) {
    return selectOption(values, null);
  }
  /**
   * This method waits for <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks, waits until
   * all specified options are present in the {@code <select>} element and selects these options.
   *
   * <p> If the target element is not a {@code <select>} element, this method throws an error. However, if the element is inside the
   * {@code <label>} element that has an associated <a
   * href="https://developer.mozilla.org/en-US/docs/Web/API/HTMLLabelElement/control">control</a>, the control will be used
   * instead.
   *
   * <p> Returns the array of option values that have been successfully selected.
   *
   * <p> Triggers a {@code change} and {@code input} event once all the provided options have been selected.
   * <pre>{@code
   * // single selection matching the value
   * handle.selectOption("blue");
   * // single selection matching the label
   * handle.selectOption(new SelectOption().setLabel("Blue"));
   * // multiple selection
   * handle.selectOption(new String[] {"red", "green", "blue"});
   * }</pre>
   *
   * @param values Options to select. If the {@code <select>} has the {@code multiple} attribute, all matching options are selected, otherwise only the
   * first option matching one of the passed options is selected. String values are equivalent to {@code {value:'string'}}. Option
   * is considered matching if all specified properties match.
   */
  List<String> selectOption(SelectOption[] values, SelectOptionOptions options);
  /**
   * This method waits for <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks, then focuses
   * the element and selects all its text content.
   */
  default void selectText() {
    selectText(null);
  }
  /**
   * This method waits for <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks, then focuses
   * the element and selects all its text content.
   */
  void selectText(SelectTextOptions options);
  /**
   * This method expects {@code elementHandle} to point to an <a
   * href="https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input">input element</a>.
   *
   * <p> Sets the value of the file input to these file paths or files. If some of the {@code filePaths} are relative paths, then they
   * are resolved relative to the the current working directory. For empty array, clears the selected files.
   */
  default void setInputFiles(Path files) {
    setInputFiles(files, null);
  }
  /**
   * This method expects {@code elementHandle} to point to an <a
   * href="https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input">input element</a>.
   *
   * <p> Sets the value of the file input to these file paths or files. If some of the {@code filePaths} are relative paths, then they
   * are resolved relative to the the current working directory. For empty array, clears the selected files.
   */
  void setInputFiles(Path files, SetInputFilesOptions options);
  /**
   * This method expects {@code elementHandle} to point to an <a
   * href="https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input">input element</a>.
   *
   * <p> Sets the value of the file input to these file paths or files. If some of the {@code filePaths} are relative paths, then they
   * are resolved relative to the the current working directory. For empty array, clears the selected files.
   */
  default void setInputFiles(Path[] files) {
    setInputFiles(files, null);
  }
  /**
   * This method expects {@code elementHandle} to point to an <a
   * href="https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input">input element</a>.
   *
   * <p> Sets the value of the file input to these file paths or files. If some of the {@code filePaths} are relative paths, then they
   * are resolved relative to the the current working directory. For empty array, clears the selected files.
   */
  void setInputFiles(Path[] files, SetInputFilesOptions options);
  /**
   * This method expects {@code elementHandle} to point to an <a
   * href="https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input">input element</a>.
   *
   * <p> Sets the value of the file input to these file paths or files. If some of the {@code filePaths} are relative paths, then they
   * are resolved relative to the the current working directory. For empty array, clears the selected files.
   */
  default void setInputFiles(FilePayload files) {
    setInputFiles(files, null);
  }
  /**
   * This method expects {@code elementHandle} to point to an <a
   * href="https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input">input element</a>.
   *
   * <p> Sets the value of the file input to these file paths or files. If some of the {@code filePaths} are relative paths, then they
   * are resolved relative to the the current working directory. For empty array, clears the selected files.
   */
  void setInputFiles(FilePayload files, SetInputFilesOptions options);
  /**
   * This method expects {@code elementHandle} to point to an <a
   * href="https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input">input element</a>.
   *
   * <p> Sets the value of the file input to these file paths or files. If some of the {@code filePaths} are relative paths, then they
   * are resolved relative to the the current working directory. For empty array, clears the selected files.
   */
  default void setInputFiles(FilePayload[] files) {
    setInputFiles(files, null);
  }
  /**
   * This method expects {@code elementHandle} to point to an <a
   * href="https://developer.mozilla.org/en-US/docs/Web/HTML/Element/input">input element</a>.
   *
   * <p> Sets the value of the file input to these file paths or files. If some of the {@code filePaths} are relative paths, then they
   * are resolved relative to the the current working directory. For empty array, clears the selected files.
   */
  void setInputFiles(FilePayload[] files, SetInputFilesOptions options);
  /**
   * This method taps the element by performing the following steps:
   * <ol>
   * <li> Wait for <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks on the element, unless
   * {@code force} option is set.</li>
   * <li> Scroll the element into view if needed.</li>
   * <li> Use {@link Page#touchscreen Page.touchscreen()} to tap the center of the element, or the specified {@code position}.</li>
   * <li> Wait for initiated navigations to either succeed or fail, unless {@code noWaitAfter} option is set.</li>
   * </ol>
   *
   * <p> If the element is detached from the DOM at any moment during the action, this method throws.
   *
   * <p> When all steps combined have not finished during the specified {@code timeout}, this method throws a {@code TimeoutError}. Passing
   * zero timeout disables this.
   *
   * <p> <strong>NOTE:</strong> {@code elementHandle.tap()} requires that the {@code hasTouch} option of the browser context be set to true.
   */
  default void tap() {
    tap(null);
  }
  /**
   * This method taps the element by performing the following steps:
   * <ol>
   * <li> Wait for <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks on the element, unless
   * {@code force} option is set.</li>
   * <li> Scroll the element into view if needed.</li>
   * <li> Use {@link Page#touchscreen Page.touchscreen()} to tap the center of the element, or the specified {@code position}.</li>
   * <li> Wait for initiated navigations to either succeed or fail, unless {@code noWaitAfter} option is set.</li>
   * </ol>
   *
   * <p> If the element is detached from the DOM at any moment during the action, this method throws.
   *
   * <p> When all steps combined have not finished during the specified {@code timeout}, this method throws a {@code TimeoutError}. Passing
   * zero timeout disables this.
   *
   * <p> <strong>NOTE:</strong> {@code elementHandle.tap()} requires that the {@code hasTouch} option of the browser context be set to true.
   */
  void tap(TapOptions options);
  /**
   * Returns the {@code node.textContent}.
   */
  String textContent();
  /**
   * Focuses the element, and then sends a {@code keydown}, {@code keypress}/{@code input}, and {@code keyup} event for each character in the text.
   *
   * <p> To press a special key, like {@code Control} or {@code ArrowDown}, use {@link ElementHandle#press ElementHandle.press()}.
   * <pre>{@code
   * elementHandle.type("Hello"); // Types instantly
   * elementHandle.type("World", new ElementHandle.TypeOptions().setDelay(100)); // Types slower, like a user
   * }</pre>
   *
   * <p> An example of typing into a text field and then submitting the form:
   * <pre>{@code
   * ElementHandle elementHandle = page.querySelector("input");
   * elementHandle.type("some text");
   * elementHandle.press("Enter");
   * }</pre>
   *
   * @param text A text to type into a focused element.
   */
  default void type(String text) {
    type(text, null);
  }
  /**
   * Focuses the element, and then sends a {@code keydown}, {@code keypress}/{@code input}, and {@code keyup} event for each character in the text.
   *
   * <p> To press a special key, like {@code Control} or {@code ArrowDown}, use {@link ElementHandle#press ElementHandle.press()}.
   * <pre>{@code
   * elementHandle.type("Hello"); // Types instantly
   * elementHandle.type("World", new ElementHandle.TypeOptions().setDelay(100)); // Types slower, like a user
   * }</pre>
   *
   * <p> An example of typing into a text field and then submitting the form:
   * <pre>{@code
   * ElementHandle elementHandle = page.querySelector("input");
   * elementHandle.type("some text");
   * elementHandle.press("Enter");
   * }</pre>
   *
   * @param text A text to type into a focused element.
   */
  void type(String text, TypeOptions options);
  /**
   * This method checks the element by performing the following steps:
   * <ol>
   * <li> Ensure that element is a checkbox or a radio input. If not, this method throws. If the element is already unchecked,
   * this method returns immediately.</li>
   * <li> Wait for <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks on the element, unless
   * {@code force} option is set.</li>
   * <li> Scroll the element into view if needed.</li>
   * <li> Use {@link Page#mouse Page.mouse()} to click in the center of the element.</li>
   * <li> Wait for initiated navigations to either succeed or fail, unless {@code noWaitAfter} option is set.</li>
   * <li> Ensure that the element is now unchecked. If not, this method throws.</li>
   * </ol>
   *
   * <p> If the element is detached from the DOM at any moment during the action, this method throws.
   *
   * <p> When all steps combined have not finished during the specified {@code timeout}, this method throws a {@code TimeoutError}. Passing
   * zero timeout disables this.
   */
  default void uncheck() {
    uncheck(null);
  }
  /**
   * This method checks the element by performing the following steps:
   * <ol>
   * <li> Ensure that element is a checkbox or a radio input. If not, this method throws. If the element is already unchecked,
   * this method returns immediately.</li>
   * <li> Wait for <a href="https://playwright.dev/java/docs/actionability/">actionability</a> checks on the element, unless
   * {@code force} option is set.</li>
   * <li> Scroll the element into view if needed.</li>
   * <li> Use {@link Page#mouse Page.mouse()} to click in the center of the element.</li>
   * <li> Wait for initiated navigations to either succeed or fail, unless {@code noWaitAfter} option is set.</li>
   * <li> Ensure that the element is now unchecked. If not, this method throws.</li>
   * </ol>
   *
   * <p> If the element is detached from the DOM at any moment during the action, this method throws.
   *
   * <p> When all steps combined have not finished during the specified {@code timeout}, this method throws a {@code TimeoutError}. Passing
   * zero timeout disables this.
   */
  void uncheck(UncheckOptions options);
  /**
   * Returns when the element satisfies the {@code state}.
   *
   * <p> Depending on the {@code state} parameter, this method waits for one of the <a
   * href="https://playwright.dev/java/docs/actionability/">actionability</a> checks to pass. This method throws when the
   * element is detached while waiting, unless waiting for the {@code "hidden"} state.
   * <ul>
   * <li> {@code "visible"} Wait until the element is <a href="https://playwright.dev/java/docs/actionability/#visible">visible</a>.</li>
   * <li> {@code "hidden"} Wait until the element is <a href="https://playwright.dev/java/docs/actionability/#visible">not visible</a>
   * or <a href="https://playwright.dev/java/docs/actionability/#attached">not attached</a>. Note that waiting for hidden
   * does not throw when the element detaches.</li>
   * <li> {@code "stable"} Wait until the element is both <a href="https://playwright.dev/java/docs/actionability/#visible">visible</a>
   * and <a href="https://playwright.dev/java/docs/actionability/#stable">stable</a>.</li>
   * <li> {@code "enabled"} Wait until the element is <a href="https://playwright.dev/java/docs/actionability/#enabled">enabled</a>.</li>
   * <li> {@code "disabled"} Wait until the element is <a href="https://playwright.dev/java/docs/actionability/#enabled">not
   * enabled</a>.</li>
   * <li> {@code "editable"} Wait until the element is <a href="https://playwright.dev/java/docs/actionability/#editable">editable</a>.</li>
   * </ul>
   *
   * <p> If the element does not satisfy the condition for the {@code timeout} milliseconds, this method will throw.
   *
   * @param state A state to wait for, see below for more details.
   */
  default void waitForElementState(ElementState state) {
    waitForElementState(state, null);
  }
  /**
   * Returns when the element satisfies the {@code state}.
   *
   * <p> Depending on the {@code state} parameter, this method waits for one of the <a
   * href="https://playwright.dev/java/docs/actionability/">actionability</a> checks to pass. This method throws when the
   * element is detached while waiting, unless waiting for the {@code "hidden"} state.
   * <ul>
   * <li> {@code "visible"} Wait until the element is <a href="https://playwright.dev/java/docs/actionability/#visible">visible</a>.</li>
   * <li> {@code "hidden"} Wait until the element is <a href="https://playwright.dev/java/docs/actionability/#visible">not visible</a>
   * or <a href="https://playwright.dev/java/docs/actionability/#attached">not attached</a>. Note that waiting for hidden
   * does not throw when the element detaches.</li>
   * <li> {@code "stable"} Wait until the element is both <a href="https://playwright.dev/java/docs/actionability/#visible">visible</a>
   * and <a href="https://playwright.dev/java/docs/actionability/#stable">stable</a>.</li>
   * <li> {@code "enabled"} Wait until the element is <a href="https://playwright.dev/java/docs/actionability/#enabled">enabled</a>.</li>
   * <li> {@code "disabled"} Wait until the element is <a href="https://playwright.dev/java/docs/actionability/#enabled">not
   * enabled</a>.</li>
   * <li> {@code "editable"} Wait until the element is <a href="https://playwright.dev/java/docs/actionability/#editable">editable</a>.</li>
   * </ul>
   *
   * <p> If the element does not satisfy the condition for the {@code timeout} milliseconds, this method will throw.
   *
   * @param state A state to wait for, see below for more details.
   */
  void waitForElementState(ElementState state, WaitForElementStateOptions options);
  /**
   * Returns element specified by selector when it satisfies {@code state} option. Returns {@code null} if waiting for {@code hidden} or
   * {@code detached}.
   *
   * <p> Wait for the {@code selector} relative to the element handle to satisfy {@code state} option (either appear/disappear from dom, or
   * become visible/hidden). If at the moment of calling the method {@code selector} already satisfies the condition, the method
   * will return immediately. If the selector doesn't satisfy the condition for the {@code timeout} milliseconds, the function will
   * throw.
   * <pre>{@code
   * page.setContent("<div><span></span></div>");
   * ElementHandle div = page.querySelector("div");
   * // Waiting for the "span" selector relative to the div.
   * ElementHandle span = div.waitForSelector("span", new ElementHandle.WaitForSelectorOptions()
   *   .setState(WaitForSelectorState.ATTACHED));
   * }</pre>
   *
   * <p> <strong>NOTE:</strong> This method does not work across navigations, use {@link Page#waitForSelector Page.waitForSelector()} instead.
   *
   * @param selector A selector to query for. See <a href="https://playwright.dev/java/docs/selectors/">working with selectors</a> for more
   * details.
   */
  default ElementHandle waitForSelector(String selector) {
    return waitForSelector(selector, null);
  }
  /**
   * Returns element specified by selector when it satisfies {@code state} option. Returns {@code null} if waiting for {@code hidden} or
   * {@code detached}.
   *
   * <p> Wait for the {@code selector} relative to the element handle to satisfy {@code state} option (either appear/disappear from dom, or
   * become visible/hidden). If at the moment of calling the method {@code selector} already satisfies the condition, the method
   * will return immediately. If the selector doesn't satisfy the condition for the {@code timeout} milliseconds, the function will
   * throw.
   * <pre>{@code
   * page.setContent("<div><span></span></div>");
   * ElementHandle div = page.querySelector("div");
   * // Waiting for the "span" selector relative to the div.
   * ElementHandle span = div.waitForSelector("span", new ElementHandle.WaitForSelectorOptions()
   *   .setState(WaitForSelectorState.ATTACHED));
   * }</pre>
   *
   * <p> <strong>NOTE:</strong> This method does not work across navigations, use {@link Page#waitForSelector Page.waitForSelector()} instead.
   *
   * @param selector A selector to query for. See <a href="https://playwright.dev/java/docs/selectors/">working with selectors</a> for more
   * details.
   */
  ElementHandle waitForSelector(String selector, WaitForSelectorOptions options);
}

