/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.hystrix;

import com.netflix.hystrix.HystrixCommand;
import com.netflix.hystrix.HystrixCommandGroupKey;
import com.netflix.hystrix.HystrixCommandKey;
import com.netflix.hystrix.HystrixCommandProperties;
import com.netflix.hystrix.HystrixEventType;
import com.netflix.hystrix.strategy.HystrixPlugins;
import com.netflix.hystrix.strategy.eventnotifier.HystrixEventNotifier;
import com.netflix.hystrix.strategy.eventnotifier.HystrixEventNotifierDefault;
import com.netflix.hystrix.util.HystrixRollingNumber;
import com.netflix.hystrix.util.HystrixRollingNumberEvent;
import com.netflix.hystrix.util.HystrixRollingPercentile;
import java.util.Collection;
import java.util.Collections;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import org.junit.Assert;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HystrixCommandMetrics {
    private static final Logger logger = LoggerFactory.getLogger(HystrixCommandMetrics.class);
    private static final ConcurrentHashMap<String, HystrixCommandMetrics> metrics = new ConcurrentHashMap();
    private final HystrixCommandProperties properties;
    private final HystrixRollingNumber counter;
    private final HystrixRollingPercentile percentileExecution;
    private final HystrixRollingPercentile percentileTotal;
    private final HystrixCommandKey key;
    private final HystrixCommandGroupKey group;
    private final AtomicInteger concurrentExecutionCount = new AtomicInteger();
    private final HystrixEventNotifier eventNotifier;
    private volatile HealthCounts healthCountsSnapshot = new HealthCounts(0L, 0L, 0);
    private volatile AtomicLong lastHealthCountsSnapshot = new AtomicLong(System.currentTimeMillis());

    public static HystrixCommandMetrics getInstance(HystrixCommandKey key, HystrixCommandGroupKey commandGroup, HystrixCommandProperties properties) {
        HystrixCommandMetrics commandMetrics = metrics.get(key.name());
        if (commandMetrics != null) {
            return commandMetrics;
        }
        commandMetrics = new HystrixCommandMetrics(key, commandGroup, properties, HystrixPlugins.getInstance().getEventNotifier());
        HystrixCommandMetrics existing = metrics.putIfAbsent(key.name(), commandMetrics);
        if (existing == null) {
            return commandMetrics;
        }
        return existing;
    }

    public static HystrixCommandMetrics getInstance(HystrixCommandKey key) {
        return metrics.get(key.name());
    }

    public static Collection<HystrixCommandMetrics> getInstances() {
        return Collections.unmodifiableCollection(metrics.values());
    }

    static void reset() {
        metrics.clear();
    }

    HystrixCommandMetrics(HystrixCommandKey key, HystrixCommandGroupKey commandGroup, HystrixCommandProperties properties, HystrixEventNotifier eventNotifier) {
        this.key = key;
        this.group = commandGroup;
        this.properties = properties;
        this.counter = new HystrixRollingNumber(properties.metricsRollingStatisticalWindowInMilliseconds(), properties.metricsRollingStatisticalWindowBuckets());
        this.percentileExecution = new HystrixRollingPercentile(properties.metricsRollingPercentileWindowInMilliseconds(), properties.metricsRollingPercentileWindowBuckets(), properties.metricsRollingPercentileBucketSize(), properties.metricsRollingPercentileEnabled());
        this.percentileTotal = new HystrixRollingPercentile(properties.metricsRollingPercentileWindowInMilliseconds(), properties.metricsRollingPercentileWindowBuckets(), properties.metricsRollingPercentileBucketSize(), properties.metricsRollingPercentileEnabled());
        this.eventNotifier = eventNotifier;
    }

    public HystrixCommandKey getCommandKey() {
        return this.key;
    }

    public HystrixCommandGroupKey getCommandGroup() {
        return this.group;
    }

    public HystrixCommandProperties getProperties() {
        return this.properties;
    }

    public long getCumulativeCount(HystrixRollingNumberEvent event) {
        return this.counter.getCumulativeSum(event);
    }

    public long getRollingCount(HystrixRollingNumberEvent event) {
        return this.counter.getRollingSum(event);
    }

    public int getExecutionTimePercentile(double percentile) {
        return this.percentileExecution.getPercentile(percentile);
    }

    public int getExecutionTimeMean() {
        return this.percentileExecution.getMean();
    }

    public int getTotalTimePercentile(double percentile) {
        return this.percentileTotal.getPercentile(percentile);
    }

    public int getTotalTimeMean() {
        return this.percentileTotal.getMean();
    }

    void resetCounter() {
        this.counter.reset();
        this.lastHealthCountsSnapshot.set(System.currentTimeMillis());
        this.healthCountsSnapshot = new HealthCounts(0L, 0L, 0);
    }

    public int getCurrentConcurrentExecutionCount() {
        return this.concurrentExecutionCount.get();
    }

    void markSuccess(long duration) {
        this.eventNotifier.markEvent(HystrixEventType.SUCCESS, this.key);
        this.counter.increment(HystrixRollingNumberEvent.SUCCESS);
    }

    void markFailure(long duration) {
        this.eventNotifier.markEvent(HystrixEventType.FAILURE, this.key);
        this.counter.increment(HystrixRollingNumberEvent.FAILURE);
    }

    void markTimeout(long duration) {
        this.eventNotifier.markEvent(HystrixEventType.TIMEOUT, this.key);
        this.counter.increment(HystrixRollingNumberEvent.TIMEOUT);
    }

    void markShortCircuited() {
        this.eventNotifier.markEvent(HystrixEventType.SHORT_CIRCUITED, this.key);
        this.counter.increment(HystrixRollingNumberEvent.SHORT_CIRCUITED);
    }

    void markThreadPoolRejection() {
        this.eventNotifier.markEvent(HystrixEventType.THREAD_POOL_REJECTED, this.key);
        this.counter.increment(HystrixRollingNumberEvent.THREAD_POOL_REJECTED);
    }

    void markSemaphoreRejection() {
        this.eventNotifier.markEvent(HystrixEventType.SEMAPHORE_REJECTED, this.key);
        this.counter.increment(HystrixRollingNumberEvent.SEMAPHORE_REJECTED);
    }

    void incrementConcurrentExecutionCount() {
        this.concurrentExecutionCount.incrementAndGet();
    }

    void decrementConcurrentExecutionCount() {
        this.concurrentExecutionCount.decrementAndGet();
    }

    void markFallbackSuccess() {
        this.eventNotifier.markEvent(HystrixEventType.FALLBACK_SUCCESS, this.key);
        this.counter.increment(HystrixRollingNumberEvent.FALLBACK_SUCCESS);
    }

    void markFallbackFailure() {
        this.eventNotifier.markEvent(HystrixEventType.FALLBACK_FAILURE, this.key);
        this.counter.increment(HystrixRollingNumberEvent.FALLBACK_FAILURE);
    }

    void markFallbackRejection() {
        this.eventNotifier.markEvent(HystrixEventType.FALLBACK_REJECTION, this.key);
        this.counter.increment(HystrixRollingNumberEvent.FALLBACK_REJECTION);
    }

    void markExceptionThrown() {
        this.eventNotifier.markEvent(HystrixEventType.EXCEPTION_THROWN, this.key);
        this.counter.increment(HystrixRollingNumberEvent.EXCEPTION_THROWN);
    }

    void markCollapsed(int numRequestsCollapsedToBatch) {
        this.eventNotifier.markEvent(HystrixEventType.COLLAPSED, this.key);
        this.counter.add(HystrixRollingNumberEvent.COLLAPSED, numRequestsCollapsedToBatch);
    }

    void markResponseFromCache() {
        this.eventNotifier.markEvent(HystrixEventType.RESPONSE_FROM_CACHE, this.key);
        this.counter.increment(HystrixRollingNumberEvent.RESPONSE_FROM_CACHE);
    }

    void addCommandExecutionTime(long duration) {
        this.percentileExecution.addValue((int)duration);
    }

    void addUserThreadExecutionTime(long duration) {
        this.percentileTotal.addValue((int)duration);
    }

    public HealthCounts getHealthCounts() {
        long lastTime = this.lastHealthCountsSnapshot.get();
        long currentTime = System.currentTimeMillis();
        if ((currentTime - lastTime >= (long)this.properties.metricsHealthSnapshotIntervalInMilliseconds().get().intValue() || this.healthCountsSnapshot == null) && this.lastHealthCountsSnapshot.compareAndSet(lastTime, currentTime)) {
            long success = this.counter.getRollingSum(HystrixRollingNumberEvent.SUCCESS);
            long failure = this.counter.getRollingSum(HystrixRollingNumberEvent.FAILURE);
            long timeout = this.counter.getRollingSum(HystrixRollingNumberEvent.TIMEOUT);
            long threadPoolRejected = this.counter.getRollingSum(HystrixRollingNumberEvent.THREAD_POOL_REJECTED);
            long semaphoreRejected = this.counter.getRollingSum(HystrixRollingNumberEvent.SEMAPHORE_REJECTED);
            long shortCircuited = this.counter.getRollingSum(HystrixRollingNumberEvent.SHORT_CIRCUITED);
            long totalCount = failure + success + timeout + threadPoolRejected + shortCircuited + semaphoreRejected;
            long errorCount = failure + timeout + threadPoolRejected + shortCircuited + semaphoreRejected;
            int errorPercentage = 0;
            if (totalCount > 0L) {
                errorPercentage = (int)((double)errorCount / (double)totalCount * 100.0);
            }
            this.healthCountsSnapshot = new HealthCounts(totalCount, errorCount, errorPercentage);
        }
        return this.healthCountsSnapshot;
    }

    public static class UnitTest {
        @Test
        public void testGetErrorPercentage() {
            try {
                HystrixCommandProperties.Setter properties = HystrixCommandProperties.Setter.getUnitTestPropertiesSetter();
                HystrixCommandMetrics metrics = UnitTest.getMetrics(properties);
                metrics.markSuccess(100L);
                Assert.assertEquals((long)0L, (long)metrics.getHealthCounts().getErrorPercentage());
                metrics.markFailure(1000L);
                Assert.assertEquals((long)50L, (long)metrics.getHealthCounts().getErrorPercentage());
                metrics.markSuccess(100L);
                metrics.markSuccess(100L);
                Assert.assertEquals((long)25L, (long)metrics.getHealthCounts().getErrorPercentage());
                metrics.markTimeout(5000L);
                metrics.markTimeout(5000L);
                Assert.assertEquals((long)50L, (long)metrics.getHealthCounts().getErrorPercentage());
                metrics.markSuccess(100L);
                metrics.markSuccess(100L);
                metrics.markSuccess(100L);
                metrics.markSuccess(5000L);
                Assert.assertEquals((long)30L, (long)metrics.getHealthCounts().getErrorPercentage());
            }
            catch (Exception e) {
                e.printStackTrace();
                Assert.fail((String)("Error occurred: " + e.getMessage()));
            }
        }

        private static HystrixCommandMetrics getMetrics(HystrixCommandProperties.Setter properties) {
            return new HystrixCommandMetrics(HystrixCommand.UnitTest.CommandKeyForUnitTest.KEY_ONE, HystrixCommand.UnitTest.CommandGroupForUnitTest.OWNER_ONE, HystrixCommandProperties.Setter.asMock(properties), HystrixEventNotifierDefault.getInstance());
        }
    }

    public static class HealthCounts {
        private final long totalCount;
        private final long errorCount;
        private final int errorPercentage;

        HealthCounts(long total, long error, int errorPercentage) {
            this.totalCount = total;
            this.errorCount = error;
            this.errorPercentage = errorPercentage;
        }

        public long getTotalRequests() {
            return this.totalCount;
        }

        public long getErrorCount() {
            return this.errorCount;
        }

        public int getErrorPercentage() {
            return this.errorPercentage;
        }
    }
}

