/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.priam.cryptography.pgp;

import com.google.inject.Inject;
import com.netflix.priam.config.IConfiguration;
import com.netflix.priam.cryptography.IFileCryptography;
import com.netflix.priam.cryptography.pgp.PgpUtil;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.NoSuchProviderException;
import java.security.Provider;
import java.security.SecureRandom;
import java.security.Security;
import java.util.Date;
import java.util.Iterator;
import org.apache.commons.io.IOUtils;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.openpgp.PGPCompressedData;
import org.bouncycastle.openpgp.PGPCompressedDataGenerator;
import org.bouncycastle.openpgp.PGPEncryptedDataGenerator;
import org.bouncycastle.openpgp.PGPEncryptedDataList;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPLiteralData;
import org.bouncycastle.openpgp.PGPLiteralDataGenerator;
import org.bouncycastle.openpgp.PGPObjectFactory;
import org.bouncycastle.openpgp.PGPOnePassSignatureList;
import org.bouncycastle.openpgp.PGPPrivateKey;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyEncryptedData;
import org.bouncycastle.openpgp.PGPSecretKey;
import org.bouncycastle.openpgp.PGPSecretKeyRingCollection;
import org.bouncycastle.openpgp.PGPUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PgpCryptography
implements IFileCryptography {
    private static final Logger logger = LoggerFactory.getLogger(PgpCryptography.class);
    private IConfiguration config;

    @Inject
    public PgpCryptography(IConfiguration config) {
        this.config = config;
    }

    private PGPSecretKeyRingCollection getPgpSecurityCollection() {
        BufferedInputStream keyIn;
        try {
            keyIn = new BufferedInputStream(new FileInputStream(this.config.getPrivateKeyLocation()));
        }
        catch (FileNotFoundException e) {
            throw new IllegalStateException("PGP private key file not found.  file: " + this.config.getPrivateKeyLocation());
        }
        try {
            return new PGPSecretKeyRingCollection(PGPUtil.getDecoderStream((InputStream)keyIn));
        }
        catch (Exception e) {
            logger.error("Exception in reading PGP security collection ring.  Msg: {}", (Object)e.getLocalizedMessage());
            throw new IllegalStateException("Exception in reading PGP security collection ring", e);
        }
    }

    private PGPPublicKey getPubKey() {
        BufferedInputStream pubKeyIS;
        try {
            pubKeyIS = new BufferedInputStream(new FileInputStream(this.config.getPgpPublicKeyLoc()));
        }
        catch (FileNotFoundException e) {
            logger.error("Exception in reading PGP security collection ring.  Msg: {}", (Object)e.getLocalizedMessage());
            throw new RuntimeException("Exception in reading PGP public key", e);
        }
        try {
            return PgpUtil.readPublicKey(pubKeyIS);
        }
        catch (Exception e) {
            throw new RuntimeException("Exception in reading & deriving the PGP public key.", e);
        }
    }

    @Override
    public InputStream decryptStream(InputStream in, char[] passwd, String objectName) throws Exception {
        logger.info("Start to decrypt object: {}", (Object)objectName);
        in = PGPUtil.getDecoderStream((InputStream)in);
        PGPObjectFactory inPgpReader = new PGPObjectFactory(in);
        Object o = inPgpReader.nextObject();
        PGPEncryptedDataList encryptedDataList = o instanceof PGPEncryptedDataList ? (PGPEncryptedDataList)o : (PGPEncryptedDataList)inPgpReader.nextObject();
        Iterator encryptedDataIterator = encryptedDataList.getEncryptedDataObjects();
        PGPPrivateKey privateKey = null;
        PGPPublicKeyEncryptedData encryptedDataStreamHandle = null;
        while (privateKey == null && encryptedDataIterator.hasNext()) {
            encryptedDataStreamHandle = (PGPPublicKeyEncryptedData)encryptedDataIterator.next();
            try {
                privateKey = PgpCryptography.findSecretKey(this.getPgpSecurityCollection(), encryptedDataStreamHandle.getKeyID(), passwd);
            }
            catch (Exception ex) {
                throw new IllegalStateException("decryption exception:  object: " + objectName + ", Exception when fetching private key using key: " + encryptedDataStreamHandle.getKeyID(), ex);
            }
        }
        if (privateKey == null) {
            throw new IllegalStateException("decryption exception:  object: " + objectName + ", Private key for message not found.");
        }
        InputStream decryptInputStream = encryptedDataStreamHandle.getDataStream(privateKey, "BC");
        PGPObjectFactory decryptedDataReader = new PGPObjectFactory(decryptInputStream);
        PGPCompressedData compressedDataReader = (PGPCompressedData)decryptedDataReader.nextObject();
        BufferedInputStream compressedStream = new BufferedInputStream(compressedDataReader.getDataStream());
        PGPObjectFactory compressedStreamReader = new PGPObjectFactory((InputStream)compressedStream);
        Object data = compressedStreamReader.nextObject();
        if (data instanceof PGPLiteralData) {
            PGPLiteralData dataPgpReader = (PGPLiteralData)data;
            return dataPgpReader.getInputStream();
        }
        if (data instanceof PGPOnePassSignatureList) {
            throw new PGPException("decryption exception:  object: " + objectName + ", encrypted data contains a signed message - not literal data.");
        }
        throw new PGPException("decryption exception:  object: " + objectName + ", data is not a simple encrypted file - type unknown.");
    }

    private static PGPPrivateKey findSecretKey(PGPSecretKeyRingCollection securityCollection, long keyID, char[] pass) throws PGPException, NoSuchProviderException {
        PGPSecretKey privateKey = securityCollection.getSecretKey(keyID);
        if (privateKey == null) {
            return null;
        }
        return privateKey.extractPrivateKey(pass, "BC");
    }

    @Override
    public Iterator<byte[]> encryptStream(InputStream is, String fileName) {
        return new ChunkEncryptorStream(is, fileName, this.getPubKey());
    }

    static {
        Security.addProvider((Provider)new BouncyCastleProvider());
    }

    public class EncryptedInputStream
    extends InputStream {
        private InputStream srcHandle;
        private ByteArrayOutputStream bos = null;
        private int bosOff = 0;
        private OutputStream pgpBosWrapper;
        private OutputStream encryptedOsWrapper;
        private PGPCompressedDataGenerator compressedDataGenerator;
        private String fileName;

        public EncryptedInputStream(InputStream is, String fileName, PGPPublicKey pubKey) {
            this.srcHandle = is;
            this.bos = new ByteArrayOutputStream();
            PGPEncryptedDataGenerator encryptedDataGenerator = new PGPEncryptedDataGenerator(3, true, new SecureRandom(), "BC");
            try {
                encryptedDataGenerator.addMethod(pubKey);
                this.pgpBosWrapper = encryptedDataGenerator.open((OutputStream)this.bos, new byte[32768]);
            }
            catch (Exception e) {
                throw new RuntimeException("Exception when wrapping PGP around our output stream", e);
            }
            this.compressedDataGenerator = new PGPCompressedDataGenerator(0);
            try {
                PGPLiteralDataGenerator literalDataGenerator = new PGPLiteralDataGenerator();
                this.encryptedOsWrapper = literalDataGenerator.open(this.compressedDataGenerator.open(this.pgpBosWrapper), 'b', fileName, new Date(), new byte[32768]);
            }
            catch (Exception e) {
                throw new RuntimeException("Exception when creating the PGP encrypted wrapper around the output stream.", e);
            }
            this.fileName = fileName;
        }

        @Override
        public synchronized int read(byte[] b, int off, int len) throws IOException {
            if (this.bosOff < this.bos.size()) {
                return this.copyToBuff(b, off, len);
            }
            this.bos.reset();
            this.bosOff = 0;
            byte[] buff = new byte[65536];
            int bytesRead = 0;
            while (this.bos.size() < len && (bytesRead = this.srcHandle.read(buff, 0, len)) > 0) {
                this.encryptedOsWrapper.write(buff, 0, bytesRead);
            }
            if (bytesRead < 0) {
                this.encryptedOsWrapper.close();
                this.compressedDataGenerator.close();
                this.pgpBosWrapper.close();
            }
            if (bytesRead < 0 && this.bos.size() == 0) {
                return bytesRead;
            }
            return this.copyToBuff(b, off, len);
        }

        private int copyToBuff(byte[] buff, int off, int len) {
            int wlen = this.bos.size() - this.bosOff < len ? this.bos.size() - this.bosOff : len;
            System.arraycopy(this.bos.toByteArray(), this.bosOff, buff, off, wlen);
            this.bosOff += wlen;
            return wlen;
        }

        @Override
        public void close() throws IOException {
            this.encryptedOsWrapper.close();
            this.compressedDataGenerator.close();
            this.pgpBosWrapper.close();
        }

        @Override
        public int read() throws IOException {
            throw new UnsupportedOperationException("Not supported, invoke read(byte[] bytes, int off, int len) instead.");
        }
    }

    public class ChunkEncryptorStream
    implements Iterator<byte[]> {
        private static final int MAX_CHUNK = 0xA00000;
        private boolean hasnext = true;
        private InputStream is;
        private InputStream encryptedSrc;
        private ByteArrayOutputStream bos;
        private BufferedOutputStream pgout;

        public ChunkEncryptorStream(InputStream is, String fileName, PGPPublicKey pubKey) {
            this.is = is;
            this.bos = new ByteArrayOutputStream();
            this.pgout = new BufferedOutputStream(this.bos);
            this.encryptedSrc = new EncryptedInputStream(this.is, fileName, pubKey);
        }

        @Override
        public boolean hasNext() {
            return this.hasnext;
        }

        @Override
        public byte[] next() {
            try {
                int count;
                byte[] buffer = new byte[2048];
                while ((count = this.encryptedSrc.read(buffer, 0, buffer.length)) != -1) {
                    this.pgout.write(buffer, 0, count);
                    if (this.bos.size() < 0xA00000) continue;
                    return this.returnSafe();
                }
                return this.done();
            }
            catch (Exception e) {
                throw new RuntimeException("Error encountered returning next chunk of ciphertext.  Msg: " + e.getLocalizedMessage(), e);
            }
        }

        @Override
        public void remove() {
        }

        private byte[] returnSafe() {
            byte[] returnData = this.bos.toByteArray();
            this.bos.reset();
            return returnData;
        }

        private byte[] done() throws IOException {
            this.pgout.flush();
            this.hasnext = false;
            byte[] returnData = this.bos.toByteArray();
            IOUtils.closeQuietly((OutputStream)this.pgout);
            IOUtils.closeQuietly((OutputStream)this.bos);
            return returnData;
        }
    }
}

