/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.priam.restore;

import com.google.common.collect.Iterators;
import com.google.common.collect.Lists;
import com.google.inject.Provider;
import com.netflix.priam.backup.AbstractBackupPath;
import com.netflix.priam.backup.BackupRestoreUtil;
import com.netflix.priam.backup.IBackupFileSystem;
import com.netflix.priam.backup.MetaData;
import com.netflix.priam.backup.Status;
import com.netflix.priam.config.IConfiguration;
import com.netflix.priam.defaultimpl.ICassandraProcess;
import com.netflix.priam.health.InstanceState;
import com.netflix.priam.identity.InstanceIdentity;
import com.netflix.priam.restore.IPostRestoreHook;
import com.netflix.priam.restore.IRestoreStrategy;
import com.netflix.priam.restore.PostRestoreHookException;
import com.netflix.priam.restore.RestoreTokenSelector;
import com.netflix.priam.scheduler.Task;
import com.netflix.priam.utils.DateUtil;
import com.netflix.priam.utils.FifoQueue;
import com.netflix.priam.utils.RetryableCallable;
import com.netflix.priam.utils.Sleeper;
import com.netflix.priam.utils.SystemUtils;
import java.io.File;
import java.io.IOException;
import java.math.BigInteger;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractRestore
extends Task
implements IRestoreStrategy {
    static final FifoQueue<AbstractBackupPath> tracker = new FifoQueue(800);
    private static final Logger logger = LoggerFactory.getLogger(AbstractRestore.class);
    private static final String JOBNAME = "AbstractRestore";
    private static final String SYSTEM_KEYSPACE = "system";
    private static BigInteger restoreToken;
    final IBackupFileSystem fs;
    final Sleeper sleeper;
    private BackupRestoreUtil backupRestoreUtil;
    private Provider<AbstractBackupPath> pathProvider;
    private InstanceIdentity id;
    private RestoreTokenSelector tokenSelector;
    private ICassandraProcess cassProcess;
    private InstanceState instanceState;
    private MetaData metaData;
    private IPostRestoreHook postRestoreHook;

    AbstractRestore(IConfiguration config, IBackupFileSystem fs, String name, Sleeper sleeper, Provider<AbstractBackupPath> pathProvider, InstanceIdentity instanceIdentity, RestoreTokenSelector tokenSelector, ICassandraProcess cassProcess, MetaData metaData, InstanceState instanceState, IPostRestoreHook postRestoreHook) {
        super(config);
        this.fs = fs;
        this.sleeper = sleeper;
        this.pathProvider = pathProvider;
        this.id = instanceIdentity;
        this.tokenSelector = tokenSelector;
        this.cassProcess = cassProcess;
        this.metaData = metaData;
        this.instanceState = instanceState;
        this.backupRestoreUtil = new BackupRestoreUtil(config.getRestoreKeyspaceFilter(), config.getRestoreCFFilter());
        this.postRestoreHook = postRestoreHook;
    }

    public static final boolean isRestoreEnabled(IConfiguration conf) {
        boolean isRestoreMode = StringUtils.isNotBlank((CharSequence)conf.getRestoreSnapshot());
        boolean isBackedupRac = CollectionUtils.isEmpty(conf.getBackupRacs()) || conf.getBackupRacs().contains(conf.getRac());
        return isRestoreMode && isBackedupRac;
    }

    private final void download(Iterator<AbstractBackupPath> fsIterator, AbstractBackupPath.BackupFileType bkupFileType) throws Exception {
        while (fsIterator.hasNext()) {
            AbstractBackupPath temp = fsIterator.next();
            if (temp.getType() == AbstractBackupPath.BackupFileType.SST && tracker.contains(temp)) continue;
            if (this.backupRestoreUtil.isFiltered(temp.getKeyspace(), temp.getColumnFamily())) {
                logger.info("Bypassing restoring file \"{}\" as it is part of the keyspace.columnfamily filter list.  Its keyspace:cf is: {}:{}", new Object[]{temp.newRestoreFile(), temp.getKeyspace(), temp.getColumnFamily()});
                continue;
            }
            if (this.config.getRestoreKeySpaces().size() != 0 && (!this.config.getRestoreKeySpaces().contains(temp.getKeyspace()) || temp.getKeyspace().equals(SYSTEM_KEYSPACE))) {
                logger.info("Bypassing restoring file \"{}\" as it is system keyspace", (Object)temp.newRestoreFile());
                continue;
            }
            if (temp.getType() != bkupFileType) continue;
            File localFileHandler = temp.newRestoreFile();
            if (logger.isDebugEnabled()) {
                logger.debug("Created local file name: " + localFileHandler.getAbsolutePath() + File.pathSeparator + localFileHandler.getName());
            }
            this.downloadFile(temp, localFileHandler);
        }
        this.waitToComplete();
    }

    private final void downloadCommitLogs(Iterator<AbstractBackupPath> fsIterator, AbstractBackupPath.BackupFileType filter, int lastN) throws Exception {
        if (fsIterator == null) {
            return;
        }
        BoundedList<AbstractBackupPath> bl = new BoundedList<AbstractBackupPath>(lastN);
        while (fsIterator.hasNext()) {
            AbstractBackupPath temp = fsIterator.next();
            if (temp.getType() == AbstractBackupPath.BackupFileType.SST && tracker.contains(temp) || temp.getType() != filter) continue;
            bl.add(temp);
        }
        this.download(bl.iterator(), filter);
    }

    private void stopCassProcess() throws IOException {
        if (this.config.getRestoreKeySpaces().size() == 0) {
            this.cassProcess.stop(true);
        }
    }

    private String getRestorePrefix() {
        String prefix = "";
        prefix = StringUtils.isNotBlank((CharSequence)this.config.getRestorePrefix()) ? this.config.getRestorePrefix() : this.config.getBackupPrefix();
        return prefix;
    }

    private final void fetchSnapshotMetaFile(String restorePrefix, List<AbstractBackupPath> out, Date startTime, Date endTime) throws IllegalStateException {
        logger.debug("Looking for snapshot meta file within restore prefix: {}", (Object)restorePrefix);
        Iterator<AbstractBackupPath> backupfiles = this.fs.list(restorePrefix, startTime, endTime);
        if (!backupfiles.hasNext()) {
            throw new IllegalStateException("meta.json not found, restore prefix: " + restorePrefix);
        }
        while (backupfiles.hasNext()) {
            AbstractBackupPath path = backupfiles.next();
            if (path.getType() != AbstractBackupPath.BackupFileType.META || !path.getFileName().equalsIgnoreCase("meta.json")) continue;
            out.add(path);
        }
    }

    @Override
    public void execute() throws Exception {
        if (!AbstractRestore.isRestoreEnabled(this.config)) {
            return;
        }
        logger.info("Starting restore for {}", (Object)this.config.getRestoreSnapshot());
        String[] restore = this.config.getRestoreSnapshot().split(",");
        AbstractBackupPath path = (AbstractBackupPath)this.pathProvider.get();
        final Date startTime = path.parseDate(restore[0]);
        final Date endTime = path.parseDate(restore[1]);
        new RetryableCallable<Void>(){

            @Override
            public Void retriableCall() throws Exception {
                logger.info("Attempting restore");
                AbstractRestore.this.restore(startTime, endTime);
                logger.info("Restore completed");
                AbstractRestore.this.sleeper.sleep(30000L);
                return null;
            }
        }.call();
    }

    public void restore(Date startTime, Date endTime) throws Exception {
        if (!this.postRestoreHook.hasValidParameters()) {
            throw new PostRestoreHookException("Invalid PostRestoreHook parameters");
        }
        this.instanceState.getRestoreStatus().resetStatus();
        this.instanceState.getRestoreStatus().setStartDateRange(DateUtil.convert(startTime));
        this.instanceState.getRestoreStatus().setEndDateRange(DateUtil.convert(endTime));
        this.instanceState.getRestoreStatus().setExecutionStartTime(LocalDateTime.now());
        this.instanceState.setRestoreStatus(Status.STARTED);
        String origToken = this.id.getInstance().getToken();
        try {
            if (this.config.isRestoreClosestToken()) {
                restoreToken = this.tokenSelector.getClosestToken(new BigInteger(origToken), startTime);
                this.id.getInstance().setToken(restoreToken.toString());
            }
            this.stopCassProcess();
            SystemUtils.cleanupDir(this.config.getDataFileLocation(), this.config.getRestoreKeySpaces());
            ArrayList metas = Lists.newArrayList();
            String prefix = this.getRestorePrefix();
            this.fetchSnapshotMetaFile(prefix, metas, startTime, endTime);
            if (metas.size() == 0) {
                logger.info("[cass_backup] No snapshot meta file found, Restore Failed.");
                this.instanceState.getRestoreStatus().setExecutionEndTime(LocalDateTime.now());
                this.instanceState.setRestoreStatus(Status.FINISHED);
                return;
            }
            Collections.sort(metas);
            AbstractBackupPath meta = (AbstractBackupPath)Iterators.getLast(metas.iterator());
            logger.info("Snapshot Meta file for restore {}", (Object)meta.getRemotePath());
            this.instanceState.getRestoreStatus().setSnapshotMetaFile(meta.getRemotePath());
            ArrayList<AbstractBackupPath> metaFile = new ArrayList<AbstractBackupPath>();
            metaFile.add(meta);
            this.download(metaFile.iterator(), AbstractBackupPath.BackupFileType.META);
            this.waitToComplete();
            List<AbstractBackupPath> snapshots = this.metaData.toJson(meta.newRestoreFile());
            this.download(snapshots.iterator(), AbstractBackupPath.BackupFileType.SNAP);
            logger.info("Downloading incrementals");
            Iterator<AbstractBackupPath> incrementals = this.fs.list(prefix, meta.getTime(), endTime);
            this.download(incrementals, AbstractBackupPath.BackupFileType.SST);
            if (this.config.isBackingUpCommitLogs()) {
                logger.info("Delete all backuped commitlog files in {}", (Object)this.config.getBackupCommitLogLocation());
                SystemUtils.cleanupDir(this.config.getBackupCommitLogLocation(), null);
                logger.info("Delete all commitlog files in {}", (Object)this.config.getCommitLogLocation());
                SystemUtils.cleanupDir(this.config.getCommitLogLocation(), null);
                Iterator<AbstractBackupPath> commitLogPathIterator = this.fs.list(prefix, meta.getTime(), endTime);
                this.downloadCommitLogs(commitLogPathIterator, AbstractBackupPath.BackupFileType.CL, this.config.maxCommitLogsRestore());
            }
            this.waitToComplete();
            logger.info("Starting post restore hook");
            this.postRestoreHook.execute();
            logger.info("Completed executing post restore hook");
            this.instanceState.getRestoreStatus().setExecutionEndTime(LocalDateTime.now());
            this.instanceState.setRestoreStatus(Status.FINISHED);
            if (!this.config.doesCassandraStartManually()) {
                this.cassProcess.start(true);
            } else {
                logger.info("config.doesCassandraStartManually() is set to True, hence Cassandra needs to be started manually ...");
            }
        }
        catch (Exception e) {
            this.instanceState.setRestoreStatus(Status.FAILED);
            this.instanceState.getRestoreStatus().setExecutionEndTime(LocalDateTime.now());
            logger.error("Error while trying to restore: {}", (Object)e.getMessage(), (Object)e);
            throw e;
        }
        finally {
            this.id.getInstance().setToken(origToken);
        }
    }

    protected abstract void downloadFile(AbstractBackupPath var1, File var2) throws Exception;

    protected abstract void waitToComplete();

    final class BoundedList<E>
    extends LinkedList<E> {
        private final int limit;

        BoundedList(int limit) {
            this.limit = limit;
        }

        @Override
        public boolean add(E o) {
            super.add(o);
            while (this.size() > this.limit) {
                super.remove();
            }
            return true;
        }
    }
}

