/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.priam.aws;

import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.BucketLifecycleConfiguration;
import com.amazonaws.services.s3.model.CompleteMultipartUploadResult;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.RateLimiter;
import com.google.inject.Provider;
import com.netflix.priam.aws.S3BackupPath;
import com.netflix.priam.aws.S3FileIterator;
import com.netflix.priam.aws.S3PartUploader;
import com.netflix.priam.aws.S3PrefixIterator;
import com.netflix.priam.backup.AbstractBackupPath;
import com.netflix.priam.backup.BackupRestoreException;
import com.netflix.priam.backup.IBackupFileSystem;
import com.netflix.priam.compress.ICompression;
import com.netflix.priam.config.IConfiguration;
import com.netflix.priam.merics.BackupMetrics;
import com.netflix.priam.notification.BackupEvent;
import com.netflix.priam.notification.BackupNotificationMgr;
import com.netflix.priam.notification.EventGenerator;
import com.netflix.priam.notification.EventObserver;
import com.netflix.priam.scheduler.BlockingSubmitThreadPoolExecutor;
import java.io.File;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class S3FileSystemBase
implements IBackupFileSystem,
EventGenerator<BackupEvent> {
    protected static final int MAX_CHUNKS = 10000;
    protected static final long MAX_BUFFERED_IN_STREAM_SIZE = 0x500000L;
    protected static final long UPLOAD_TIMEOUT = 0x6DDD00L;
    private static final Logger logger = LoggerFactory.getLogger(S3FileSystemBase.class);
    protected AtomicLong bytesUploaded = new AtomicLong();
    protected AtomicLong bytesDownloaded = new AtomicLong();
    protected BackupMetrics backupMetrics;
    protected AmazonS3 s3Client;
    protected IConfiguration config;
    protected Provider<AbstractBackupPath> pathProvider;
    protected ICompression compress;
    protected BlockingSubmitThreadPoolExecutor executor;
    protected RateLimiter rateLimiter;
    private final CopyOnWriteArrayList<EventObserver<BackupEvent>> observers = new CopyOnWriteArrayList();

    public S3FileSystemBase(Provider<AbstractBackupPath> pathProvider, ICompression compress, IConfiguration config, BackupMetrics backupMetrics, BackupNotificationMgr backupNotificationMgr) {
        this.pathProvider = pathProvider;
        this.compress = compress;
        this.config = config;
        this.backupMetrics = backupMetrics;
        int threads = config.getMaxBackupUploadThreads();
        LinkedBlockingQueue<Runnable> queue = new LinkedBlockingQueue<Runnable>(threads);
        this.executor = new BlockingSubmitThreadPoolExecutor(threads, queue, 0x6DDD00L);
        double throttleLimit = config.getUploadThrottle();
        this.rateLimiter = RateLimiter.create((double)(throttleLimit < 1.0 ? Double.MAX_VALUE : throttleLimit));
        this.addObserver(backupNotificationMgr);
    }

    public AmazonS3 getS3Client() {
        return this.s3Client;
    }

    public void setS3Client(AmazonS3 client) {
        this.s3Client = client;
    }

    protected String getPrefix(IConfiguration config) {
        String prefix = StringUtils.isNotBlank((CharSequence)config.getRestorePrefix()) ? config.getRestorePrefix() : config.getBackupPrefix();
        String[] paths = prefix.split(String.valueOf(S3BackupPath.PATH_SEP));
        return paths[0];
    }

    @Override
    public void cleanup() {
        List rules;
        AmazonS3 s3Client = this.getS3Client();
        String clusterPath = ((AbstractBackupPath)this.pathProvider.get()).clusterPrefix("");
        logger.debug("Bucket: {}", (Object)this.config.getBackupPrefix());
        BucketLifecycleConfiguration lifeConfig = s3Client.getBucketLifecycleConfiguration(this.config.getBackupPrefix());
        logger.debug("Got bucket:{} lifecycle.{}", (Object)this.config.getBackupPrefix(), (Object)lifeConfig);
        if (lifeConfig == null) {
            lifeConfig = new BucketLifecycleConfiguration();
            rules = Lists.newArrayList();
            lifeConfig.setRules(rules);
        }
        if (this.updateLifecycleRule(this.config, rules = lifeConfig.getRules(), clusterPath)) {
            if (rules.size() > 0) {
                lifeConfig.setRules(rules);
                s3Client.setBucketLifecycleConfiguration(this.config.getBackupPrefix(), lifeConfig);
            } else {
                s3Client.deleteBucketLifecycleConfiguration(this.config.getBackupPrefix());
            }
        }
    }

    private boolean updateLifecycleRule(IConfiguration config, List<BucketLifecycleConfiguration.Rule> rules, String prefix) {
        BucketLifecycleConfiguration.Rule rule = null;
        for (BucketLifecycleConfiguration.Rule lcRule : rules) {
            if (!lcRule.getPrefix().equals(prefix)) continue;
            rule = lcRule;
            break;
        }
        if (rule == null && config.getBackupRetentionDays() <= 0) {
            return false;
        }
        if (rule != null && rule.getExpirationInDays() == config.getBackupRetentionDays()) {
            logger.info("Cleanup rule already set");
            return false;
        }
        if (rule == null) {
            rule = new BucketLifecycleConfiguration.Rule().withExpirationInDays(config.getBackupRetentionDays()).withPrefix(prefix);
            rule.setStatus("Enabled");
            rule.setId(prefix);
            rules.add(rule);
            logger.info("Setting cleanup for {} to {} days", (Object)rule.getPrefix(), (Object)rule.getExpirationInDays());
        } else if (config.getBackupRetentionDays() > 0) {
            logger.info("Setting cleanup for {} to {} days", (Object)rule.getPrefix(), (Object)config.getBackupRetentionDays());
            rule.setExpirationInDays(config.getBackupRetentionDays());
        } else {
            logger.info("Removing cleanup rule for {}", (Object)rule.getPrefix());
            rules.remove(rule);
        }
        return true;
    }

    private void postProcessingPerFile(AbstractBackupPath path, long startTimeInMilliSecs, long completedTimeInMilliSecs) {
        try {
            long sizeInBytes = path.getSize();
            long elapseTimeInMillisecs = completedTimeInMilliSecs - startTimeInMilliSecs;
            long elapseTimeInSecs = elapseTimeInMillisecs / 1000L;
            long bytesReadPerSec = 0L;
            Double speedInKBps = 0.0;
            if (elapseTimeInSecs > 0L && sizeInBytes > 0L) {
                bytesReadPerSec = sizeInBytes / elapseTimeInSecs;
                speedInKBps = (double)bytesReadPerSec / 1024.0;
            } else {
                bytesReadPerSec = sizeInBytes;
                speedInKBps = sizeInBytes;
            }
            logger.info("Upload rate for file: {}, elapsse time in sec(s): {}, KB per sec: {}", new Object[]{path.getFileName(), elapseTimeInSecs, speedInKBps});
            this.backupMetrics.recordUploadRate(sizeInBytes);
        }
        catch (Exception e) {
            logger.error("Post processing of file {} failed, not fatal.", (Object)path.getFileName(), (Object)e);
        }
    }

    protected void reinitialize() {
        this.bytesUploaded = new AtomicLong(0L);
    }

    protected void logDiagnosticInfo(AbstractBackupPath fileUploaded, CompleteMultipartUploadResult res) {
        File f = fileUploaded.getBackupFile();
        String fName = f.getAbsolutePath();
        logger.info("Uploaded file: {}, object eTag: {}", (Object)fName, (Object)res.getETag());
    }

    @Override
    public void upload(AbstractBackupPath path, InputStream in) throws BackupRestoreException {
        this.reinitialize();
        long chunkSize = this.config.getBackupChunkSize();
        if (path.getSize() > 0L) {
            chunkSize = path.getSize() / chunkSize >= 10000L ? path.getSize() / 9999L : chunkSize;
        }
        logger.info("Uploading to {}/{} with chunk size {}", new Object[]{this.config.getBackupPrefix(), path.getRemotePath(), chunkSize});
        long startTime = System.nanoTime();
        this.notifyEventStart(new BackupEvent(path));
        this.uploadFile(path, in, chunkSize);
        long completedTime = System.nanoTime();
        this.postProcessingPerFile(path, TimeUnit.NANOSECONDS.toMillis(startTime), TimeUnit.NANOSECONDS.toMillis(completedTime));
        this.notifyEventSuccess(new BackupEvent(path));
        this.backupMetrics.incrementValidUploads();
    }

    protected void checkSuccessfulUpload(CompleteMultipartUploadResult resultS3MultiPartUploadComplete, AbstractBackupPath path) throws BackupRestoreException {
        if (null == resultS3MultiPartUploadComplete || null == resultS3MultiPartUploadComplete.getETag()) {
            this.backupMetrics.incrementInvalidUploads();
            throw new BackupRestoreException("Error uploading file as ETag or CompleteMultipartUploadResult is NULL -" + path.getFileName());
        }
        String eTagObjectId = resultS3MultiPartUploadComplete.getETag();
        this.logDiagnosticInfo(path, resultS3MultiPartUploadComplete);
    }

    protected BackupRestoreException encounterError(AbstractBackupPath path, S3PartUploader s3PartUploader, Exception e) {
        s3PartUploader.abortUpload();
        return this.encounterError(path, e);
    }

    protected BackupRestoreException encounterError(AbstractBackupPath path, Exception e) {
        AmazonS3Exception a;
        String amazoneErrorCode;
        this.backupMetrics.incrementInvalidUploads();
        if (e instanceof AmazonS3Exception && (amazoneErrorCode = (a = (AmazonS3Exception)e).getErrorCode()) != null && !amazoneErrorCode.isEmpty() && amazoneErrorCode.equalsIgnoreCase("slowdown")) {
            this.backupMetrics.incrementAwsSlowDownException(1);
            logger.warn("Received slow down from AWS when uploading file: {}", (Object)path.getFileName());
        }
        logger.error("Error uploading file {}, a datapart was not uploaded.", (Object)path.getFileName(), (Object)e);
        this.notifyEventFailure(new BackupEvent(path));
        return new BackupRestoreException("Error uploading file " + path.getFileName(), e);
    }

    abstract void uploadFile(AbstractBackupPath var1, InputStream var2, long var3) throws BackupRestoreException;

    @Override
    public void download(AbstractBackupPath path, OutputStream os, String filePath) throws BackupRestoreException {
        try {
            this.download(path, os);
        }
        catch (Exception e) {
            throw new BackupRestoreException(e.getMessage(), e);
        }
    }

    @Override
    public void download(AbstractBackupPath path, OutputStream os) throws BackupRestoreException {
        logger.info("Downloading {} from S3 bucket {}", (Object)path.getRemotePath(), (Object)this.getPrefix(this.config));
        long contentLen = this.s3Client.getObjectMetadata(this.getPrefix(this.config), path.getRemotePath()).getContentLength();
        path.setSize(contentLen);
        try {
            this.downloadFile(path, os);
            this.bytesDownloaded.addAndGet(contentLen);
            this.backupMetrics.incrementValidDownloads();
        }
        catch (BackupRestoreException e) {
            this.backupMetrics.incrementInvalidDownloads();
            throw e;
        }
    }

    protected abstract void downloadFile(AbstractBackupPath var1, OutputStream var2) throws BackupRestoreException;

    @Override
    public long getBytesUploaded() {
        return this.bytesUploaded.get();
    }

    @Override
    public long getAWSSlowDownExceptionCounter() {
        return this.backupMetrics.getAwsSlowDownException();
    }

    public long downloadCount() {
        return this.backupMetrics.getValidDownloads();
    }

    public long uploadCount() {
        return this.backupMetrics.getValidUploads();
    }

    @Override
    public void shutdown() {
        if (this.executor != null) {
            this.executor.shutdown();
        }
    }

    @Override
    public Iterator<AbstractBackupPath> listPrefixes(Date date) {
        return new S3PrefixIterator(this.config, this.pathProvider, this.s3Client, date);
    }

    @Override
    public Iterator<AbstractBackupPath> list(String path, Date start, Date till) {
        return new S3FileIterator(this.pathProvider, this.s3Client, path, start, till);
    }

    @Override
    public final void addObserver(EventObserver<BackupEvent> observer) {
        if (observer == null) {
            throw new NullPointerException("observer must not be null.");
        }
        this.observers.addIfAbsent(observer);
    }

    @Override
    public void removeObserver(EventObserver<BackupEvent> observer) {
        if (observer == null) {
            throw new NullPointerException("observer must not be null.");
        }
        this.observers.remove(observer);
    }

    @Override
    public void notifyEventStart(BackupEvent event) {
        this.observers.forEach((Consumer<EventObserver<BackupEvent>>)((Consumer<EventObserver>)eventObserver -> eventObserver.updateEventStart(event)));
    }

    @Override
    public void notifyEventSuccess(BackupEvent event) {
        this.observers.forEach((Consumer<EventObserver<BackupEvent>>)((Consumer<EventObserver>)eventObserver -> eventObserver.updateEventSuccess(event)));
    }

    @Override
    public void notifyEventFailure(BackupEvent event) {
        this.observers.forEach((Consumer<EventObserver<BackupEvent>>)((Consumer<EventObserver>)eventObserver -> eventObserver.updateEventFailure(event)));
    }

    @Override
    public void notifyEventStop(BackupEvent event) {
        this.observers.forEach((Consumer<EventObserver<BackupEvent>>)((Consumer<EventObserver>)eventObserver -> eventObserver.updateEventStop(event)));
    }
}

