/*
 * Copyright (c) 2000, 2020, Oracle and/or its affiliates.
 *
 * Licensed under the Universal Permissive License v 1.0 as shown at
 * http://oss.oracle.com/licenses/upl.
 */


package com.tangosol.dev.component;


import com.tangosol.util.Base;
import com.tangosol.util.StringTable;


/**
* Contains information on what child classes must be created and what their
* supers will be.
*
* @version 1.00, 10/29/99
* @author  Cameron Purdy
*/
public class CompilePlan
        extends    Base
        implements Constants
    {
    /**
    * Construct a CompilePlan.
    *
    * Example inheritance tree:
    *
    *   Component
    *     Window
    *       W2
    *         W3
    *
    * Example of tblSuper in which this component (W3) contains two child
    * components that will exist at run time, the first (OK) will either
    * produce a class that inherits from the corresponding child (W2$OK)
    * on the super of W3 (W2) or, if OK can be discarded at this level,
    * that class (W2$OK) will be used at runtime instead of the discarded
    * W3$OK.  The second (Cancel) will either inherit from Window$Cancel
    * (implying that W2$Cancel has no delta) or if W3$Cancel can be
    * discarded, then Window$Cancel will be used at runtime instead of
    * W3$Cancel.
    *
    *   Key     Value
    *   ------  -----------------------
    *   OK      Component.Window.W2$OK
    *   Cancel  Component.Window$Cancel
    *
    *
    * Example of tblDelta in which the Cancel class can be optimized out:
    *
    *   Key
    *   ---
    *   OK
    *
    * @param sName     a String value that specifies this component's fully
    *                  qualified name (e.g. Component.Window.W2.W3)
    * @param tblSuper  a map that is keyed by local component names with
    *                  the corresponding value being fully qualified
    *                  name of the "runtime super"
    * @param tblDelta  a map that is keyed by local component names that
    *                  cannot be discarded because they have deltas
    */
    public CompilePlan(String sName, StringTable tblSuper, StringTable tblDelta)
        {
        m_sName    = sName;
        m_tblSuper = tblSuper;
        m_tblDelta = tblDelta;
        }

    /**
    * Get the name of the global component for which this compile plan exists.
    *
    * @return fully qualified name of the component for which this
    *         compile plan exists
    */
    public String getGlobalName()
        {
        return m_sName;
        }

    /**
    * Determine whether a class must be generated for a child component with
    * the specified fully qualified name.
    * (For a global component, the class will always be generated, thus
    * this method must return false)
    *
    * Note that a non-optimizing build that does not discard any child
    * classes would not use this method.
    *
    * @param  sName  a fully qualified component name
    *
    * @return true is the class doesnt have to be generated
    */
    public boolean isDiscardable(String sName)
        {
        String sLocalName = Component.getLocalName(sName);

        if (sLocalName == null)
            {
            return false;
            }

        if (!m_tblSuper.contains(sLocalName))
            {
            throw new IllegalArgumentException("No such child:  " + sName);
            }

        // classes from child components are discardable iff the child
        // does not have a delta
        return !m_tblDelta.contains(sLocalName);
        }

    /**
    * Determine the qualified name (e.g. Component.Window$OK) of the super
    * component that is known to result in a class being generated that is a
    * super class of the class that would be generated by this child component.
    *
    * This method has two purposes:
    *   1)  If this child is discardable, and as a result the compilation
    *       does not produce a class from this child, then the class that
    *       contains this child must instantiate the class corresponding
    *       to the super component name returned;
    *   2)  Otherwise, if this child is not discarded, it must inherit from
    *       the class corresponding to the super component name returned.
    *
    * Note that a non-optimizing build that does not discard any child
    * classes would not use this method.
    *
    * @param  sName  a fully qualified component name
    *
    * @return  the qualified name of the component that will result in a
    * `        class that the class resulting from this component will
    *          inherit from
    */
    public String getSuperName(String sName)
        {
        if (sName.indexOf(LOCAL_ID_DELIM) < 0)
            {
            // global super is determinable directly from the name
            int of = sName.lastIndexOf(GLOBAL_ID_DELIM);
            if (of < 0)
                {
                // inherits from "root super"
                return "";
                }
            return sName.substring(0, of);
            }
        else
            {
            String sLocalName = Component.getLocalName(sName);

            if (!m_tblSuper.contains(sLocalName))
                {
                throw new IllegalArgumentException("No such child:  " + sName);
                }

            // local super is specified in the map of child to super
            return (String) m_tblSuper.get(sLocalName);
            }
        }


    // ----- data members ---------------------------------------------------

    /**
    * A String value that specifies this component's fully qualified name
    * (e.g. Component.Window.W2.W3).
    */
    private String m_sName;

    /**
    * A map that is keyed by local component names (A, A$B, etc.) with the
    * corresponding value being fully qualified name of the "runtime
    * super".
    */
    private StringTable m_tblSuper;

    /**
    * A set that is keyed by local component names (A, A$B, etc.) that cannot
    * be discarded because they have deltas.
    */
    private StringTable m_tblDelta;
    }