////////////////////////////////////////////////////////////////////////////////
// checkstyle: Checks Java source code for adherence to a set of rules.
// Copyright (C) 2001-2015 the original author or authors.
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
////////////////////////////////////////////////////////////////////////////////

package com.puppycrawl.tools.checkstyle.checks.indentation;

import com.puppycrawl.tools.checkstyle.api.DetailAST;
import com.puppycrawl.tools.checkstyle.api.TokenTypes;

/**
 * Handler for parents of blocks ('if', 'else', 'while', etc).
 * <P>
 * The "block" handler classes use a common superclass BlockParentHandler,
 * employing the Template Method pattern.
 * </P>
 *
 * <UL>
 *   <LI>template method to get the lcurly</LI>
 *   <LI>template method to get the rcurly</LI>
 *   <LI>if curlys aren't present, then template method to get expressions
 *       is called</LI>
 *   <LI>now all the repetitous code which checks for BOL, if curlys are on
 *       same line, etc. can be collapsed into the superclass</LI>
 * </UL>
 *
 *
 * @author jrichard
 */
public class BlockParentHandler extends AbstractExpressionHandler {
    /**
     * Children checked by parent handlers.
     */
    private static final int[] CHECKED_CHILDREN = new int[] {
        TokenTypes.VARIABLE_DEF,
        TokenTypes.EXPR,
        TokenTypes.OBJBLOCK,
        TokenTypes.LITERAL_BREAK,
        TokenTypes.LITERAL_RETURN,
        TokenTypes.LITERAL_THROW,
        TokenTypes.LITERAL_CONTINUE,
    };

    /**
     * Construct an instance of this handler with the given indentation check,
     * name, abstract syntax tree, and parent handler.
     *
     * @param indentCheck   the indentation check
     * @param name          the name of the handler
     * @param ast           the abstract syntax tree
     * @param parent        the parent handler
     */
    public BlockParentHandler(IndentationCheck indentCheck,
        String name, DetailAST ast, AbstractExpressionHandler parent) {
        super(indentCheck, name, ast, parent);
    }

    /**
     * Returns array of token types which should be checked among childrens.
     * @return array of token types to check.
     */
    protected int[] getCheckedChildren() {
        return CHECKED_CHILDREN.clone();
    }

    /**
     * Get the top level expression being managed by this handler.
     *
     * @return the top level expression
     */
    protected DetailAST getToplevelAST() {
        return getMainAst();
    }

    /**
     * Check the indent of the top level token.
     */
    protected void checkToplevelToken() {
        final DetailAST toplevel = getToplevelAST();

        if (toplevel == null
            || getLevel().accept(expandedTabsColumnNo(toplevel)) || hasLabelBefore()) {
            return;
        }
        if (!toplevelMustStartLine() && !startsLine(toplevel)) {
            return;
        }
        logError(toplevel, "", expandedTabsColumnNo(toplevel));
    }

    /**
     * Check if the top level token has label before.
     * @return true if the top level token has label before.
     */
    protected boolean hasLabelBefore() {
        final DetailAST parent = getToplevelAST().getParent();
        return parent != null && parent.getType() == TokenTypes.LABELED_STAT
            && parent.getLineNo() == getToplevelAST().getLineNo();
    }

    /**
     * Determines if the top level token must start the line.
     *
     * @return true
     */
    protected boolean toplevelMustStartLine() {
        return true;
    }

    /**
     * Determines if this block expression has curly braces.
     *
     * @return true if curly braces are present, false otherwise
     */
    protected boolean hasCurlys() {
        return getLCurly() != null && getRCurly() != null;
    }

    /**
     * Get the left curly brace portion of the expression we are handling.
     *
     * @return the left curly brace expression
     */
    protected DetailAST getLCurly() {
        return getMainAst().findFirstToken(TokenTypes.SLIST);
    }

    /**
     * Get the right curly brace portion of the expression we are handling.
     *
     * @return the right curly brace expression
     */
    protected DetailAST getRCurly() {
        final DetailAST slist = getMainAst().findFirstToken(TokenTypes.SLIST);
        if (slist == null) {
            return null;
        }

        return slist.findFirstToken(TokenTypes.RCURLY);
    }

    /**
     * Check the indentation of the left curly brace.
     */
    protected void checkLCurly() {
        // the lcurly can either be at the correct indentation, or nested
        // with a previous expression
        final DetailAST lcurly = getLCurly();
        final int lcurlyPos = expandedTabsColumnNo(lcurly);

        if (curlyLevel().accept(lcurlyPos) || !startsLine(lcurly)) {
            return;
        }

        logError(lcurly, "lcurly", lcurlyPos);
    }

    /**
     * Get the expected indentation level for the curly braces.
     *
     * @return the curly brace indentation level
     */
    protected IndentLevel curlyLevel() {
        return new IndentLevel(getLevel(), getBraceAdjustment());
    }

    /**
     * Determines if the right curly brace must be at the start of the line.
     *
     * @return true
     */
    protected boolean rcurlyMustStart() {
        return true;
    }

    /**
     * Determines if child elements within the expression may be nested.
     *
     * @return false
     */
    protected boolean childrenMayNest() {
        return false;
    }

    /**
     * Check the indentation of the right curly brace.
     */
    protected void checkRCurly() {
        // the rcurly can either be at the correct indentation, or
        // on the same line as the lcurly
        final DetailAST lcurly = getLCurly();
        final DetailAST rcurly = getRCurly();
        final int rcurlyPos = expandedTabsColumnNo(rcurly);

        if (curlyLevel().accept(rcurlyPos)
            || !rcurlyMustStart() && !startsLine(rcurly)
            || areOnSameLine(rcurly, lcurly)) {
            return;
        }
        logError(rcurly, "rcurly", rcurlyPos, curlyLevel());
    }

    /**
     * Get the child element that is not a list of statements.
     *
     * @return the non-list child element
     */
    protected DetailAST getNonlistChild() {
        return getMainAst().findFirstToken(TokenTypes.RPAREN).getNextSibling();
    }

    /**
     * Check the indentation level of a child that is not a list of statements.
     */
    private void checkNonlistChild() {
        final DetailAST nonlist = getNonlistChild();
        if (nonlist == null) {
            return;
        }

        final IndentLevel expected = new IndentLevel(getLevel(), getBasicOffset());
        checkExpressionSubtree(nonlist, expected, false, false);
    }

    /**
     * Get the child element representing the list of statements.
     *
     * @return the statement list child
     */
    protected DetailAST getListChild() {
        return getMainAst().findFirstToken(TokenTypes.SLIST);
    }

    /**
     * Get the right parenthesis portion of the expression we are handling.
     *
     * @return the right parenthis expression
     */
    protected DetailAST getRParen() {
        return getMainAst().findFirstToken(TokenTypes.RPAREN);
    }

    /**
     * Get the left parenthesis portion of the expression we are handling.
     *
     * @return the left parenthis expression
     */
    protected DetailAST getLParen() {
        return getMainAst().findFirstToken(TokenTypes.LPAREN);
    }

    @Override
    public void checkIndentation() {
        checkToplevelToken();
        // seperate to allow for eventual configuration
        checkLParen(getLParen());
        checkRParen(getLParen(), getRParen());
        if (hasCurlys()) {
            checkLCurly();
            checkRCurly();
        }
        final DetailAST listChild = getListChild();
        if (listChild != null) {
            // NOTE: switch statements usually don't have curlys
            if (!hasCurlys() || !areOnSameLine(getLCurly(), getRCurly())) {
                checkChildren(listChild,
                              getCheckedChildren(),
                              getChildrenExpectedLevel(),
                              true,
                              childrenMayNest());
            }
        }
        else {
            checkNonlistChild();
        }
    }

    /**
     * @return indentation level expected for children
     */
    protected IndentLevel getChildrenExpectedLevel() {
        // if we have multileveled expected level then we should
        // try to suggest single level to children using curlies'
        // levels.
        if (getLevel().isMultiLevel() && hasCurlys()
            && !areOnSameLine(getLCurly(), getRCurly())) {
            if (startsLine(getLCurly())) {
                return new IndentLevel(expandedTabsColumnNo(getLCurly()) + getBasicOffset());
            }
            else if (startsLine(getRCurly())) {
                final IndentLevel level = new IndentLevel(curlyLevel(), getBasicOffset());
                level.addAcceptedIndent(level.getFirstIndentLevel() + getLineWrappingIndent());
                return level;
            }
        }
        return new IndentLevel(getLevel(), getBasicOffset());
    }

    @Override
    public IndentLevel suggestedChildLevel(AbstractExpressionHandler child) {
        return getChildrenExpectedLevel();
    }

    /**
     * A shortcut for <code>IndentationCheck</code> property.
     * @return value of lineWrappingIndentation property
     *         of <code>IndentationCheck</code>
     */
    private int getLineWrappingIndent() {
        return getIndentCheck().getLineWrappingIndentation();
    }
}
