/*
 * Decompiled with CFR 0.152.
 */
package com.unboundid.asn1;

import com.unboundid.asn1.ASN1Element;
import com.unboundid.asn1.ASN1Exception;
import com.unboundid.asn1.ASN1GeneralizedTime;
import com.unboundid.asn1.ASN1Messages;
import com.unboundid.asn1.ASN1StreamReaderSequence;
import com.unboundid.asn1.ASN1StreamReaderSet;
import com.unboundid.asn1.ASN1UTCTime;
import com.unboundid.util.Debug;
import com.unboundid.util.Mutable;
import com.unboundid.util.NotNull;
import com.unboundid.util.Nullable;
import com.unboundid.util.StaticUtils;
import com.unboundid.util.ThreadSafety;
import com.unboundid.util.ThreadSafetyLevel;
import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.net.SocketTimeoutException;
import java.util.Date;
import java.util.logging.Level;
import javax.security.sasl.SaslClient;

@Mutable
@ThreadSafety(level=ThreadSafetyLevel.NOT_THREADSAFE)
public final class ASN1StreamReader
implements Closeable {
    private boolean ignoreInitialSocketTimeout;
    private boolean ignoreSubsequentSocketTimeout;
    @Nullable
    private volatile ByteArrayInputStream saslInputStream;
    @NotNull
    private final InputStream inputStream;
    private final int maxElementSize;
    private long totalBytesRead;
    @Nullable
    private volatile SaslClient saslClient;

    public ASN1StreamReader(@NotNull InputStream inputStream) {
        this(inputStream, Integer.MAX_VALUE);
    }

    public ASN1StreamReader(@NotNull InputStream inputStream, int maxElementSize) {
        this.inputStream = inputStream.markSupported() ? inputStream : new BufferedInputStream(inputStream);
        this.maxElementSize = maxElementSize > 0 ? maxElementSize : Integer.MAX_VALUE;
        this.totalBytesRead = 0L;
        this.ignoreInitialSocketTimeout = false;
        this.ignoreSubsequentSocketTimeout = false;
        this.saslClient = null;
        this.saslInputStream = null;
    }

    @Override
    public void close() throws IOException {
        this.inputStream.close();
    }

    long getTotalBytesRead() {
        return this.totalBytesRead;
    }

    @Deprecated
    public boolean ignoreSocketTimeoutException() {
        return this.ignoreInitialSocketTimeout;
    }

    public boolean ignoreInitialSocketTimeoutException() {
        return this.ignoreInitialSocketTimeout;
    }

    public boolean ignoreSubsequentSocketTimeoutException() {
        return this.ignoreSubsequentSocketTimeout;
    }

    @Deprecated
    public void setIgnoreSocketTimeout(boolean ignoreSocketTimeout) {
        this.ignoreInitialSocketTimeout = ignoreSocketTimeout;
        this.ignoreSubsequentSocketTimeout = ignoreSocketTimeout;
    }

    public void setIgnoreSocketTimeout(boolean ignoreInitialSocketTimeout, boolean ignoreSubsequentSocketTimeout) {
        this.ignoreInitialSocketTimeout = ignoreInitialSocketTimeout;
        this.ignoreSubsequentSocketTimeout = ignoreSubsequentSocketTimeout;
    }

    public int peek() throws IOException {
        InputStream is;
        if (this.saslClient == null) {
            is = this.inputStream;
        } else {
            if (this.saslInputStream == null || this.saslInputStream.available() <= 0) {
                this.readAndDecodeSASLData(-1);
            }
            is = this.saslInputStream;
        }
        is.mark(1);
        int byteRead = this.read(true);
        is.reset();
        return byteRead;
    }

    private int readType() throws IOException {
        int typeInt = this.read(true);
        if (typeInt < 0) {
            this.close();
        } else {
            ++this.totalBytesRead;
        }
        return typeInt;
    }

    private int readLength() throws IOException {
        int length = this.read(false);
        if (length < 0) {
            throw new IOException(ASN1Messages.ERR_READ_END_BEFORE_FIRST_LENGTH.get());
        }
        ++this.totalBytesRead;
        if (length > 127) {
            int numLengthBytes = length & 0x7F;
            length = 0;
            if (numLengthBytes < 1 || numLengthBytes > 4) {
                throw new IOException(ASN1Messages.ERR_READ_LENGTH_TOO_LONG.get(numLengthBytes));
            }
            for (int i = 0; i < numLengthBytes; ++i) {
                int lengthInt = this.read(false);
                if (lengthInt < 0) {
                    throw new IOException(ASN1Messages.ERR_READ_END_BEFORE_LENGTH_END.get());
                }
                length <<= 8;
                length |= lengthInt & 0xFF;
            }
            this.totalBytesRead += (long)numLengthBytes;
        }
        if (length < 0 || this.maxElementSize > 0 && length > this.maxElementSize) {
            throw new IOException(ASN1Messages.ERR_READ_LENGTH_EXCEEDS_MAX.get(length, this.maxElementSize));
        }
        return length;
    }

    private void skip(int numBytes) throws IOException {
        if (numBytes <= 0) {
            return;
        }
        if (this.saslClient != null) {
            int bytesRead;
            int skippedSoFar = 0;
            byte[] skipBuffer = new byte[numBytes];
            do {
                if ((bytesRead = this.read(skipBuffer, skippedSoFar, numBytes - skippedSoFar)) < 0) {
                    return;
                }
                this.totalBytesRead += (long)bytesRead;
            } while ((skippedSoFar += bytesRead) < numBytes);
            return;
        }
        long totalBytesSkipped = this.inputStream.skip(numBytes);
        while (totalBytesSkipped < (long)numBytes) {
            long bytesSkipped = this.inputStream.skip((long)numBytes - totalBytesSkipped);
            if (bytesSkipped <= 0L) {
                while (totalBytesSkipped < (long)numBytes) {
                    int byteRead = this.read(false);
                    if (byteRead < 0) {
                        throw new IOException(ASN1Messages.ERR_READ_END_BEFORE_VALUE_END.get());
                    }
                    ++totalBytesSkipped;
                }
                continue;
            }
            totalBytesSkipped += bytesSkipped;
        }
        this.totalBytesRead += (long)numBytes;
    }

    @Nullable
    public ASN1Element readElement() throws IOException {
        int type = this.readType();
        if (type < 0) {
            return null;
        }
        int length = this.readLength();
        int valueBytesRead = 0;
        int bytesRemaining = length;
        byte[] value = new byte[length];
        while (valueBytesRead < length) {
            int bytesRead = this.read(value, valueBytesRead, bytesRemaining);
            if (bytesRead < 0) {
                throw new IOException(ASN1Messages.ERR_READ_END_BEFORE_VALUE_END.get());
            }
            valueBytesRead += bytesRead;
            bytesRemaining -= bytesRead;
        }
        this.totalBytesRead += (long)length;
        ASN1Element e = new ASN1Element((byte)type, value);
        Debug.debugASN1Read(e);
        return e;
    }

    @Nullable
    public Boolean readBoolean() throws IOException, ASN1Exception {
        int type = this.readType();
        if (type < 0) {
            return null;
        }
        int length = this.readLength();
        if (length == 1) {
            int value = this.read(false);
            if (value < 0) {
                throw new IOException(ASN1Messages.ERR_READ_END_BEFORE_VALUE_END.get());
            }
            ++this.totalBytesRead;
            Boolean booleanValue = value != 0;
            Debug.debugASN1Read(Level.INFO, "Boolean", type, 1, booleanValue);
            return booleanValue;
        }
        this.skip(length);
        throw new ASN1Exception(ASN1Messages.ERR_BOOLEAN_INVALID_LENGTH.get());
    }

    @Nullable
    public Integer readEnumerated() throws IOException, ASN1Exception {
        return this.readInteger();
    }

    @Nullable
    public Date readGeneralizedTime() throws IOException, ASN1Exception {
        int type = this.readType();
        if (type < 0) {
            return null;
        }
        int length = this.readLength();
        int valueBytesRead = 0;
        int bytesRemaining = length;
        byte[] value = new byte[length];
        while (valueBytesRead < length) {
            int bytesRead = this.read(value, valueBytesRead, bytesRemaining);
            if (bytesRead < 0) {
                throw new IOException(ASN1Messages.ERR_READ_END_BEFORE_VALUE_END.get());
            }
            valueBytesRead += bytesRead;
            bytesRemaining -= bytesRead;
        }
        this.totalBytesRead += (long)length;
        String timestamp = StaticUtils.toUTF8String(value);
        Date date = new Date(ASN1GeneralizedTime.decodeTimestamp(timestamp));
        Debug.debugASN1Read(Level.INFO, "GeneralizedTime", type, length, timestamp);
        return date;
    }

    @Nullable
    public Integer readInteger() throws IOException, ASN1Exception {
        int type = this.readType();
        if (type < 0) {
            return null;
        }
        int length = this.readLength();
        if (length == 0 || length > 4) {
            this.skip(length);
            throw new ASN1Exception(ASN1Messages.ERR_INTEGER_INVALID_LENGTH.get(length));
        }
        boolean negative = false;
        int intValue = 0;
        for (int i = 0; i < length; ++i) {
            int byteRead = this.read(false);
            if (byteRead < 0) {
                throw new IOException(ASN1Messages.ERR_READ_END_BEFORE_VALUE_END.get());
            }
            if (i == 0) {
                negative = (byteRead & 0x80) != 0;
            }
            intValue <<= 8;
            intValue |= byteRead & 0xFF;
        }
        if (negative) {
            switch (length) {
                case 1: {
                    intValue |= 0xFFFFFF00;
                    break;
                }
                case 2: {
                    intValue |= 0xFFFF0000;
                    break;
                }
                case 3: {
                    intValue |= 0xFF000000;
                }
            }
        }
        this.totalBytesRead += (long)length;
        Debug.debugASN1Read(Level.INFO, "Integer", type, length, intValue);
        return intValue;
    }

    @Nullable
    public Long readLong() throws IOException, ASN1Exception {
        int type = this.readType();
        if (type < 0) {
            return null;
        }
        int length = this.readLength();
        if (length == 0 || length > 8) {
            this.skip(length);
            throw new ASN1Exception(ASN1Messages.ERR_LONG_INVALID_LENGTH.get(length));
        }
        boolean negative = false;
        long longValue = 0L;
        for (int i = 0; i < length; ++i) {
            int byteRead = this.read(false);
            if (byteRead < 0) {
                throw new IOException(ASN1Messages.ERR_READ_END_BEFORE_VALUE_END.get());
            }
            if (i == 0) {
                negative = (byteRead & 0x80) != 0;
            }
            longValue <<= 8;
            longValue |= (long)byteRead & 0xFFL;
        }
        if (negative) {
            switch (length) {
                case 1: {
                    longValue |= 0xFFFFFFFFFFFFFF00L;
                    break;
                }
                case 2: {
                    longValue |= 0xFFFFFFFFFFFF0000L;
                    break;
                }
                case 3: {
                    longValue |= 0xFFFFFFFFFF000000L;
                    break;
                }
                case 4: {
                    longValue |= 0xFFFFFFFF00000000L;
                    break;
                }
                case 5: {
                    longValue |= 0xFFFFFF0000000000L;
                    break;
                }
                case 6: {
                    longValue |= 0xFFFF000000000000L;
                    break;
                }
                case 7: {
                    longValue |= 0xFF00000000000000L;
                }
            }
        }
        this.totalBytesRead += (long)length;
        Debug.debugASN1Read(Level.INFO, "Long", type, length, longValue);
        return longValue;
    }

    @Nullable
    public BigInteger readBigInteger() throws IOException, ASN1Exception {
        int type = this.readType();
        if (type < 0) {
            return null;
        }
        int length = this.readLength();
        if (length == 0) {
            throw new ASN1Exception(ASN1Messages.ERR_BIG_INTEGER_DECODE_EMPTY_VALUE.get());
        }
        byte[] valueBytes = new byte[length];
        for (int i = 0; i < length; ++i) {
            int byteRead = this.read(false);
            if (byteRead < 0) {
                throw new IOException(ASN1Messages.ERR_READ_END_BEFORE_VALUE_END.get());
            }
            valueBytes[i] = (byte)byteRead;
        }
        BigInteger bigIntegerValue = new BigInteger(valueBytes);
        this.totalBytesRead += (long)length;
        Debug.debugASN1Read(Level.INFO, "BigInteger", type, length, bigIntegerValue);
        return bigIntegerValue;
    }

    public void readNull() throws IOException, ASN1Exception {
        int type = this.readType();
        if (type < 0) {
            return;
        }
        int length = this.readLength();
        if (length != 0) {
            this.skip(length);
            throw new ASN1Exception(ASN1Messages.ERR_NULL_HAS_VALUE.get());
        }
        Debug.debugASN1Read(Level.INFO, "Null", type, 0, null);
    }

    @Nullable
    public byte[] readBytes() throws IOException {
        int type = this.readType();
        if (type < 0) {
            return null;
        }
        int length = this.readLength();
        int valueBytesRead = 0;
        int bytesRemaining = length;
        byte[] value = new byte[length];
        while (valueBytesRead < length) {
            int bytesRead = this.read(value, valueBytesRead, bytesRemaining);
            if (bytesRead < 0) {
                throw new IOException(ASN1Messages.ERR_READ_END_BEFORE_VALUE_END.get());
            }
            valueBytesRead += bytesRead;
            bytesRemaining -= bytesRead;
        }
        this.totalBytesRead += (long)length;
        Debug.debugASN1Read(Level.INFO, "byte[]", type, length, value);
        return value;
    }

    @Nullable
    public String readString() throws IOException {
        int type = this.readType();
        if (type < 0) {
            return null;
        }
        int length = this.readLength();
        int valueBytesRead = 0;
        int bytesRemaining = length;
        byte[] value = new byte[length];
        while (valueBytesRead < length) {
            int bytesRead = this.read(value, valueBytesRead, bytesRemaining);
            if (bytesRead < 0) {
                throw new IOException(ASN1Messages.ERR_READ_END_BEFORE_VALUE_END.get());
            }
            valueBytesRead += bytesRead;
            bytesRemaining -= bytesRead;
        }
        this.totalBytesRead += (long)length;
        String s = StaticUtils.toUTF8String(value);
        Debug.debugASN1Read(Level.INFO, "String", type, length, s);
        return s;
    }

    @Nullable
    public Date readUTCTime() throws IOException, ASN1Exception {
        int type = this.readType();
        if (type < 0) {
            return null;
        }
        int length = this.readLength();
        int valueBytesRead = 0;
        int bytesRemaining = length;
        byte[] value = new byte[length];
        while (valueBytesRead < length) {
            int bytesRead = this.read(value, valueBytesRead, bytesRemaining);
            if (bytesRead < 0) {
                throw new IOException(ASN1Messages.ERR_READ_END_BEFORE_VALUE_END.get());
            }
            valueBytesRead += bytesRead;
            bytesRemaining -= bytesRead;
        }
        this.totalBytesRead += (long)length;
        String timestamp = StaticUtils.toUTF8String(value);
        Date date = new Date(ASN1UTCTime.decodeTimestamp(timestamp));
        Debug.debugASN1Read(Level.INFO, "UTCTime", type, length, timestamp);
        return date;
    }

    @Nullable
    public ASN1StreamReaderSequence beginSequence() throws IOException {
        int type = this.readType();
        if (type < 0) {
            return null;
        }
        int length = this.readLength();
        Debug.debugASN1Read(Level.INFO, "Sequence Header", type, length, null);
        return new ASN1StreamReaderSequence(this, (byte)type, length);
    }

    @Nullable
    public ASN1StreamReaderSet beginSet() throws IOException {
        int type = this.readType();
        if (type < 0) {
            return null;
        }
        int length = this.readLength();
        Debug.debugASN1Read(Level.INFO, "Set Header", type, length, null);
        return new ASN1StreamReaderSet(this, (byte)type, length);
    }

    private int read(boolean initial) throws IOException {
        if (this.saslClient != null) {
            int b;
            if (this.saslInputStream != null && (b = this.saslInputStream.read()) >= 0) {
                return b;
            }
            this.readAndDecodeSASLData(-1);
            return this.saslInputStream.read();
        }
        try {
            int b = this.inputStream.read();
            if (this.saslClient == null || b < 0) {
                return b;
            }
            this.readAndDecodeSASLData(b);
            return this.saslInputStream.read();
        }
        catch (SocketTimeoutException ste) {
            Debug.debugException(Level.FINEST, ste);
            if (initial && this.ignoreInitialSocketTimeout || !initial && this.ignoreSubsequentSocketTimeout) {
                while (true) {
                    try {
                        return this.inputStream.read();
                    }
                    catch (SocketTimeoutException ste2) {
                        Debug.debugException(Level.FINEST, ste2);
                        continue;
                    }
                    break;
                }
            }
            throw ste;
        }
    }

    private int read(@NotNull byte[] buffer, int offset, int length) throws IOException {
        if (this.saslClient != null) {
            int bytesRead;
            if (this.saslInputStream != null && (bytesRead = this.saslInputStream.read(buffer, offset, length)) > 0) {
                return bytesRead;
            }
            this.readAndDecodeSASLData(-1);
            return this.saslInputStream.read(buffer, offset, length);
        }
        try {
            return this.inputStream.read(buffer, offset, length);
        }
        catch (SocketTimeoutException ste) {
            Debug.debugException(Level.FINEST, ste);
            if (this.ignoreSubsequentSocketTimeout) {
                while (true) {
                    try {
                        return this.inputStream.read(buffer, offset, length);
                    }
                    catch (SocketTimeoutException ste2) {
                        Debug.debugException(Level.FINEST, ste2);
                        continue;
                    }
                    break;
                }
            }
            throw ste;
        }
    }

    void setSASLClient(@NotNull SaslClient saslClient) {
        this.saslClient = saslClient;
    }

    private void readAndDecodeSASLData(int firstByte) throws IOException {
        int numBytesRead;
        int numWrappedBytes = 0;
        int numLengthBytes = 4;
        if (firstByte >= 0) {
            numLengthBytes = 3;
            numWrappedBytes = firstByte;
        }
        for (int i = 0; i < numLengthBytes; ++i) {
            int b = this.inputStream.read();
            if (b < 0) {
                if (i == 0 && firstByte < 0) {
                    this.saslInputStream = new ByteArrayInputStream(StaticUtils.NO_BYTES);
                    continue;
                }
                throw new IOException(ASN1Messages.ERR_STREAM_READER_EOS_READING_SASL_LENGTH.get(i));
            }
            numWrappedBytes = numWrappedBytes << 8 | b & 0xFF;
        }
        if (this.maxElementSize > 0 && numWrappedBytes > this.maxElementSize) {
            throw new IOException(ASN1Messages.ERR_READ_SASL_LENGTH_EXCEEDS_MAX.get(numWrappedBytes, this.maxElementSize));
        }
        int wrappedDataPos = 0;
        byte[] wrappedData = new byte[numWrappedBytes];
        do {
            if ((numBytesRead = this.inputStream.read(wrappedData, wrappedDataPos, numWrappedBytes - wrappedDataPos)) >= 0) continue;
            throw new IOException(ASN1Messages.ERR_STREAM_READER_EOS_READING_SASL_DATA.get(wrappedDataPos, numWrappedBytes));
        } while ((wrappedDataPos += numBytesRead) < numWrappedBytes);
        byte[] unwrappedData = this.saslClient.unwrap(wrappedData, 0, numWrappedBytes);
        this.saslInputStream = new ByteArrayInputStream(unwrappedData, 0, unwrappedData.length);
    }
}

