/*
 * Copyright 2006-2008 Web Cohesion
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.webcohesion.enunciate.modules.ruby_json_client;

import com.webcohesion.enunciate.api.datatype.DataTypeReference;
import com.webcohesion.enunciate.api.resources.Entity;
import com.webcohesion.enunciate.api.resources.MediaTypeDescriptor;
import com.webcohesion.enunciate.javac.decorations.TypeMirrorDecorator;
import com.webcohesion.enunciate.javac.decorations.type.DecoratedTypeMirror;
import com.webcohesion.enunciate.metadata.ClientName;
import com.webcohesion.enunciate.modules.jackson.EnunciateJacksonContext;
import com.webcohesion.enunciate.modules.jackson.api.impl.DataTypeReferenceImpl;
import com.webcohesion.enunciate.modules.jackson.model.adapters.Adaptable;
import com.webcohesion.enunciate.modules.jackson.model.adapters.AdapterType;
import com.webcohesion.enunciate.modules.jackson.model.types.JsonClassType;
import com.webcohesion.enunciate.modules.jackson.model.types.JsonType;
import com.webcohesion.enunciate.modules.jackson.model.util.JacksonUtil;
import com.webcohesion.enunciate.modules.jackson1.EnunciateJackson1Context;
import com.webcohesion.enunciate.util.HasClientConvertibleType;
import freemarker.template.TemplateModelException;

import javax.activation.DataHandler;
import javax.lang.model.element.ElementKind;
import javax.lang.model.element.TypeElement;
import javax.lang.model.type.*;
import javax.xml.datatype.XMLGregorianCalendar;
import javax.xml.namespace.QName;
import java.net.URI;
import java.sql.Timestamp;
import java.util.*;

import static com.webcohesion.enunciate.javac.decorations.element.ElementUtils.isCollection;
import static com.webcohesion.enunciate.javac.decorations.element.ElementUtils.isMap;

/**
 * Conversion from java types to Ruby types.
 *
 * @author Ryan Heaton
 */
public class ClientClassnameForMethod extends com.webcohesion.enunciate.util.freemarker.ClientClassnameForMethod {

  private final Map<String, String> classConversions = new HashMap<String, String>();
  private final EnunciateJacksonContext jacksonContext;
  private final EnunciateJackson1Context jackson1Context;

  public ClientClassnameForMethod(Map<String, String> conversions, EnunciateJacksonContext jacksonContext, EnunciateJackson1Context jackson1Context) {
    super(conversions, jacksonContext == null ? jackson1Context.getContext() : jacksonContext.getContext());
    this.jacksonContext = jacksonContext;
    this.jackson1Context = jackson1Context;

    classConversions.put(Boolean.class.getName(), "Boolean");
    classConversions.put(String.class.getName(), "String");
    classConversions.put(Integer.class.getName(), "Fixnum");
    classConversions.put(Short.class.getName(), "Fixnum");
    classConversions.put(Byte.class.getName(), "Fixnum");
    classConversions.put(Double.class.getName(), "Float");
    classConversions.put(Long.class.getName(), "Bignum");
    classConversions.put(java.math.BigInteger.class.getName(), "Bignum");
    classConversions.put(java.math.BigDecimal.class.getName(), "Float");
    classConversions.put(Float.class.getName(), "Float");
    classConversions.put(Character.class.getName(), "Fixnum");
    classConversions.put(Date.class.getName(), "Time");
    classConversions.put(Timestamp.class.getName(), "Time");
    classConversions.put(DataHandler.class.getName(), "String");
    classConversions.put(java.awt.Image.class.getName(), "String");
    classConversions.put(javax.xml.transform.Source.class.getName(), "String");
    classConversions.put(QName.class.getName(), "String");
    classConversions.put(URI.class.getName(), "String");
    classConversions.put(UUID.class.getName(), "String");
    classConversions.put(XMLGregorianCalendar.class.getName(), "String");
    classConversions.put(GregorianCalendar.class.getName(), "Time");
    classConversions.put(Calendar.class.getName(), "Time");
    classConversions.put(javax.xml.datatype.Duration.class.getName(), "String");
    classConversions.put(javax.xml.bind.JAXBElement.class.getName(), "Object");
    classConversions.put(Object.class.getName(), "Object");
  }

  @Override
  public String convertUnwrappedObject(Object unwrapped) throws TemplateModelException {
    if (unwrapped instanceof Entity) {
      List<? extends MediaTypeDescriptor> mediaTypes = ((Entity) unwrapped).getMediaTypes();
      for (MediaTypeDescriptor mediaType : mediaTypes) {
        if (EnunciateJacksonContext.SYNTAX_LABEL.equals(mediaType.getSyntax())) {
          DataTypeReference dataType = mediaType.getDataType();
          if (dataType instanceof DataTypeReferenceImpl) {
            JsonType xmlType = ((DataTypeReferenceImpl) dataType).getJsonType();
            if (xmlType instanceof JsonClassType) {
              super.convertUnwrappedObject(((JsonClassType) xmlType).getTypeDefinition());
            }
          }

          if (dataType instanceof com.webcohesion.enunciate.modules.jackson1.api.impl.DataTypeReferenceImpl) {
            com.webcohesion.enunciate.modules.jackson1.model.types.JsonType jsonType = ((com.webcohesion.enunciate.modules.jackson1.api.impl.DataTypeReferenceImpl) dataType).getJsonType();
            if (jsonType instanceof com.webcohesion.enunciate.modules.jackson1.model.types.JsonClassType) {
              super.convertUnwrappedObject(((com.webcohesion.enunciate.modules.jackson1.model.types.JsonClassType) jsonType).getTypeDefinition());
            }
          }
        }
      }

      return "Object";
    }

    return super.convertUnwrappedObject(unwrapped);
  }

  @Override
  public String convert(TypeElement declaration) throws TemplateModelException {
    String fqn = declaration.getQualifiedName().toString();
    if (classConversions.containsKey(fqn)) {
      return classConversions.get(fqn);
    }
    else if (declaration.getKind() == ElementKind.ENUM) {
      return "String";
    }
    else if (isCollection(declaration)) {
      return "Array";
    }
    else if (isMap(declaration)) {
      return "Hash";
    }

    if (this.jacksonContext != null) {
      AdapterType adapterType = JacksonUtil.findAdapterType(declaration, this.jacksonContext);
      if (adapterType != null) {
        return convert(adapterType.getAdaptingType());
      }
    }

    if (this.jackson1Context != null) {
      com.webcohesion.enunciate.modules.jackson1.model.adapters.AdapterType adapter1Type = com.webcohesion.enunciate.modules.jackson1.model.util.JacksonUtil.findAdapterType(declaration, this.jackson1Context);
      if (adapter1Type != null) {
        return convert(adapter1Type.getAdaptingType());
      }
    }

    String convertedPackage = convertPackage(this.context.getProcessingEnvironment().getElementUtils().getPackageOf(declaration));
    ClientName specifiedName = declaration.getAnnotation(ClientName.class);
    String simpleName = specifiedName == null ? declaration.getSimpleName().toString() : specifiedName.value();
    return convertedPackage + getPackageSeparator() + simpleName;
  }

  @Override
  public String convert(HasClientConvertibleType element) throws TemplateModelException {
    if (element instanceof Adaptable && ((Adaptable) element).isAdapted()) {
      return convert(((Adaptable) element).getAdapterType().getAdaptingType((DecoratedTypeMirror) element.getClientConvertibleType(), this.context));
    }

    if (element instanceof com.webcohesion.enunciate.modules.jackson1.model.adapters.Adaptable && ((com.webcohesion.enunciate.modules.jackson1.model.adapters.Adaptable) element).isAdapted()) {
      return convert(((com.webcohesion.enunciate.modules.jackson1.model.adapters.Adaptable) element).getAdapterType().getAdaptingType((DecoratedTypeMirror) element.getClientConvertibleType(), this.context));
    }

    return super.convert(element);
  }

  @Override
  public String convert(TypeMirror typeMirror) throws TemplateModelException {
    DecoratedTypeMirror decorated = (DecoratedTypeMirror) TypeMirrorDecorator.decorate(typeMirror, this.context.getProcessingEnvironment());
    if (decorated.isPrimitive()) {
      TypeKind kind = decorated.getKind();
      switch (kind) {
        case BOOLEAN:
          return "Boolean";
        case BYTE:
        case INT:
        case SHORT:
        case CHAR:
          return "Fixnum";
        case FLOAT:
        case DOUBLE:
          return "Float";
        case LONG:
          return "Bignum";
        default:
          return "String";
      }
    }
    else if (decorated.isEnum()) {
      return "String";
    }
    else if (decorated.isCollection()) {
      return "Array";
    }
    else if (decorated.isArray()) {
      TypeMirror componentType = ((ArrayType) decorated).getComponentType();
      if ((componentType instanceof PrimitiveType) && componentType.getKind() == TypeKind.BYTE) {
        return "String";
      }
    }

    return super.convert(typeMirror);
  }

  @Override
  public String convertDeclaredTypeArguments(List<? extends TypeMirror> actualTypeArguments) throws TemplateModelException {
    return ""; //we'll handle generics ourselves.
  }

  @Override
  public String convert(TypeVariable typeVariable) throws TemplateModelException {
    String conversion = "Object";

    if (typeVariable.getUpperBound() != null) {
      conversion = convert(typeVariable.getUpperBound());
    }

    return conversion;
  }

  @Override
  protected String getPackageSeparator() {
    return "::";
  }
}