/**
 * (C) Copyright 2011-2015 FastConnect SAS
 * (http://www.fastconnect.fr/) and others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package fr.fastconnect.factory.tibco.bw.maven.packaging;

import java.io.IOException;
import java.math.BigInteger;
import java.util.Enumeration;
import java.util.List;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.bind.JAXBException;

import org.apache.commons.io.FileUtils;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;

import com.tibco.xmlns.applicationmanagement.ApplicationType;
import com.tibco.xmlns.applicationmanagement.EncodingType;
import com.tibco.xmlns.applicationmanagement.Events;
import com.tibco.xmlns.applicationmanagement.HttpRepoInstance;
import com.tibco.xmlns.applicationmanagement.LocalRepoInstance;
import com.tibco.xmlns.applicationmanagement.RepoInstances;
import com.tibco.xmlns.applicationmanagement.RepoType;
import com.tibco.xmlns.applicationmanagement.RvRepoInstance;

import fr.fastconnect.factory.tibco.bw.maven.packaging.monitoring.FailureEvent;
import fr.fastconnect.factory.tibco.bw.maven.packaging.monitoring.LogEvent;

/**
 * <p>
 * This goal recreates an XML Deployment Descriptor file from flat properties
 * files generated by
 * <a href="./generate-properties-from-xml-mojo.html">
 * bw:generate-properties-from-xml</a>.<br />
 * Any manual modification performed on the properties files will be integrated
 * in the final XML file.
 * </p>
 * <p>
 * A JAXB implementation of the TIBCO ApplicationManagement schema
 * (namespace = "http://www.tibco.com/xmlns/ApplicationManagement")
 * will be used.<br />
 * Refer to {@link ApplicationManagement} class for details.
 * </p>
 * 
 * @author Mathieu Debove
 *
 */
@Mojo( name="generate-xml-from-properties",
defaultPhase=LifecyclePhase.PREPARE_PACKAGE )
public class GenerateXMLFromPropertiesMojo extends AbstractPackagingMojo {

	protected static final String FLAT_PATH_SEPARATOR = "/";

	protected static final String APPLICATION_MANAGEMENT_COPY_FAILURE = "Failed to copy the Application Management file from";
	protected static final String APPLICATION_MANAGEMENT_LOAD_FAILURE = "Failed to load the Application Management file in";
	protected static final String PROPERTIES_GLOBAL_VARIABLES_LOAD_FAILURE = "Failed to load the Global Variables Properties file in";
	protected static final String PROPERTIES_SERVICES_LOAD_FAILURE = "Failed to load the Services Properties file in";
	protected static final String APPLICATION_MANAGEMENT_MERGE_FAILURE = "Failed to merge the Properties files in the Application Management file";
	protected static final String APPLICATION_MANAGEMENT_MERGE_SUCCESS = "Successfully merged the Properties in the final XML Deployment Descriptor in";

	private ApplicationManagement application;

	@Override
	protected String getArtifactFileExtension() {
		return XML_EXTENSION;
	}

	/**
	 * <p>
	 * First we will copy the XML file extracted from the EAR to the final file
	 * to be produced after merging all the properties files.
	 * </p>
	 * 
	 * <p>
	 * Then we will initialize the {@link ApplicationType} by unmarshalling
	 * this final file.
	 * </p>
	 * 
	 * @throws MojoExecutionException
	 */
	private void init() throws MojoExecutionException {
		try {
			FileUtils.copyFile(deploymentDescriptor, deploymentDescriptorFinal);
		} catch (IOException e) {
			throw new MojoExecutionException(APPLICATION_MANAGEMENT_COPY_FAILURE + " '" + deploymentDescriptor + "' to '" + deploymentDescriptorFinal + "'", e);
		}
		try {
			application = new ApplicationManagement(deploymentDescriptorFinal);
		} catch (JAXBException e) {
			throw new MojoExecutionException(APPLICATION_MANAGEMENT_LOAD_FAILURE + " '" + deploymentDescriptorFinal + "'", e);
		}
	}

	/**
	 * <p>
	 * This will merge the Global Variables properties file into the
	 * {@link ApplicationType} object.
	 * </p>
	 * 
	 * @throws MojoExecutionException
	 */
	private void mergeGlobalVariables() throws MojoExecutionException {
		Properties propertiesGlobalVariables;
		try {
			propertiesGlobalVariables = loadPropertiesFile(deploymentGlobalVariables);
		} catch (Exception e) {
			throw new MojoExecutionException(PROPERTIES_GLOBAL_VARIABLES_LOAD_FAILURE + " '" + deploymentGlobalVariables + "'", e);
		}
		
		Enumeration<Object> e = propertiesGlobalVariables.keys();
   		while (e.hasMoreElements()) {
			String key = (String) e.nextElement();
			String value = propertiesGlobalVariables.getProperty(key);

			application.setGlobalVariable(key, value);
		}
	}

	private void mergeServices() throws MojoExecutionException {
		Properties propertiesServices;
		try {
			propertiesServices = loadPropertiesFile(deploymentServices);
		} catch (Exception e) {
			throw new MojoExecutionException(PROPERTIES_SERVICES_LOAD_FAILURE + " '" + deploymentServices + "'", e);
		}
		
		Pattern patternElements = Pattern.compile("(\\w+(\\[[\\w -\\.\\/?]*\\])?)+");

		Enumeration<Object> e = propertiesServices.keys();
   		while (e.hasMoreElements()) {
			String currentPath = "";
			
			String key = (String) e.nextElement();
			String value = propertiesServices.getProperty(key);
			
			Matcher matcherElements;
			Object parent = null;
			while ((matcherElements = patternElements.matcher(key)).find()) {
				String element = matcherElements.group();
				currentPath = currentPath + FLAT_PATH_SEPARATOR + element;
				
				parent = application.getElement(currentPath, element, value, parent);
				
				if (key.equals(element)) {
					break;
				}
				key = key.substring(element.length()+1);
			}
		}
		
		try {
			application.removeDefaultBindingIfNotExists(loadPropertiesFile(deploymentServices));
		} catch (Exception ex) {
			throw new MojoExecutionException(PROPERTIES_SERVICES_LOAD_FAILURE + " '" + deploymentServices + "'", ex);
		}
		application.removeDuplicateBinding();
	}

	/**
	 * This will update the &lt;repoInstances> element.
	 */
	private void updateRepoInstances() {
		RepoInstances repoInstances = application.getRepoInstances();
		
		RvRepoInstance rvRepoInstance = repoInstances.getRvRepoInstance();
		rvRepoInstance.setDiscoveryTimout(BigInteger.valueOf(repoRvDiscoveryTimeout));
		rvRepoInstance.setTimeout(BigInteger.valueOf(repoRvTimeout));
		rvRepoInstance.setDaemon(repoRvDaemon);
		rvRepoInstance.setService(repoRvService);
		rvRepoInstance.setNetwork(repoRvNetwork);
		rvRepoInstance.setRegionalSubject(repoRvRegionalSubject);
		rvRepoInstance.setOperationRetry(BigInteger.valueOf(repoRvOperationRetry));
		rvRepoInstance.setExtraPropertyFile(repoRvExtraPropertyFile);
		rvRepoInstance.setServer(repoRvServer);
		rvRepoInstance.setUser(repoRvUser);
		rvRepoInstance.setPassword(repoRvPassword);
		
		HttpRepoInstance httpRepoInstance = repoInstances.getHttpRepoInstance();
		httpRepoInstance.setTimeout(BigInteger.valueOf(repoHttpTimeout));
		httpRepoInstance.setUrl(repoHttpUrl);
		httpRepoInstance.setServer(repoHttpServer);
		httpRepoInstance.setUser(repoHttpUser);
		httpRepoInstance.setPassword(repoHttpPassword);
		httpRepoInstance.setExtraPropertyFile(repoHttpExtraPropertyFile);
		
		LocalRepoInstance localRepoInstance = repoInstances.getLocalRepoInstance();
		EncodingType encoding;
		try {
			encoding = EncodingType.valueOf(repoLocalEncoding);
		} catch (IllegalArgumentException e) {
			encoding = EncodingType.UTF_8;
		}
		localRepoInstance.setEncoding(encoding);

		repoInstances.setRvRepoInstance(rvRepoInstance);
		repoInstances.setHttpRepoInstance(httpRepoInstance);
		repoInstances.setLocalRepoInstance(localRepoInstance);
		
		RepoType repoType;
		try {
			repoType = RepoType.valueOf(repoSelectInstance.toUpperCase());
		} catch (IllegalArgumentException e) {
			repoType = RepoType.LOCAL;
		}
		repoInstances.setSelected(repoType);
	}

	private void updateMonitoringRules() {
		Events events = createMonitoringEvents();
		application.addMonitoringEventsToAllServices(events);
	}

	private void updateAdditionalInfo() {
		getLog().debug("Updating additional information...");
		getLog().debug("MaxDeploymentRevision: " + maxDeploymentRevision);
		getLog().debug("Contact: " + contact);
		getLog().debug("Description: " + description);
		application.setMaxDeploymentRevision(maxDeploymentRevision);
		application.setContact(contact);
		application.setDescription(description);
	}

	public void execute() throws MojoExecutionException {
		if (super.skip()) {
           	if (deploymentDescriptorFinal != null && !deploymentDescriptorFinal.exists() && touchFinalDeploymentDescriptorIfSkipped) {
           		// final deployment descriptor was not created because packaging is skipped
           		// however we "touch" the final deployment descriptor file so there is an empty final deployment descriptor file created
           		try {
           			deploymentDescriptorFinal.getParentFile().mkdirs();
           			deploymentDescriptorFinal.createNewFile();
				} catch (IOException e) {
					throw new MojoExecutionException(e.getLocalizedMessage(), e);
				}
           	}

			if (deploymentDescriptorFinal != null && deploymentDescriptorFinal.exists()) {
				attachFile(deploymentDescriptorFinal, XML_TYPE, "final");
			} else {
            	getLog().warn(WARN_NO_ARTIFACT_ATTACHED);
			}

			return;
		}

		init();

		mergeGlobalVariables();
		mergeServices();

		updateRepoInstances();
		updateMonitoringRules();
		updateAdditionalInfo();

		try {
			application.save();
		} catch (JAXBException e) {
			throw new MojoExecutionException(APPLICATION_MANAGEMENT_MERGE_FAILURE + " '" + deploymentServices + "'", e);
		}

		attachFile(deploymentDescriptorFinal, XML_TYPE, "final");

		getLog().info(APPLICATION_MANAGEMENT_MERGE_SUCCESS + " '" + deploymentDescriptorFinal + "'");
	}

	private Events createMonitoringEvents() {
		Events result = new Events();

		if (events != null) {
			List<FailureEvent> failures = events.getFailures();
			if (failures != null) {
				for (FailureEvent failureEvent : failures) {
					ApplicationManagement.addEvent(result, failureEvent);
				}
			}
			List<LogEvent> logs = events.getLogs();
			if (logs != null) {
				for (LogEvent logEvent : logs) {
					ApplicationManagement.addEvent(result, logEvent);
				}
			}
		}

		return result;
	}

}
