/*
 * Decompiled with CFR 0.152.
 */
package io.cellery.security.cell.sts.server.core.service;

import com.nimbusds.jwt.JWTClaimsSet;
import com.nimbusds.jwt.PlainJWT;
import com.nimbusds.jwt.SignedJWT;
import io.cellery.security.cell.sts.server.authorization.AuthorizationFailedException;
import io.cellery.security.cell.sts.server.authorization.AuthorizationService;
import io.cellery.security.cell.sts.server.core.CellStsUtils;
import io.cellery.security.cell.sts.server.core.STSTokenGenerator;
import io.cellery.security.cell.sts.server.core.context.store.UserContextStore;
import io.cellery.security.cell.sts.server.core.exception.CellSTSRequestValidationFailedException;
import io.cellery.security.cell.sts.server.core.exception.TokenValidationFailureException;
import io.cellery.security.cell.sts.server.core.model.CellStsRequest;
import io.cellery.security.cell.sts.server.core.model.CellStsResponse;
import io.cellery.security.cell.sts.server.core.model.RequestDestination;
import io.cellery.security.cell.sts.server.core.model.config.CellStsConfiguration;
import io.cellery.security.cell.sts.server.core.service.CelleryCellSTSException;
import io.cellery.security.cell.sts.server.core.validators.CellSTSRequestValidator;
import io.cellery.security.cell.sts.server.core.validators.CelleryHostnameVerifier;
import io.cellery.security.cell.sts.server.core.validators.CelleryTrustManager;
import io.cellery.security.cell.sts.server.core.validators.DefaultCellSTSReqValidator;
import io.cellery.security.cell.sts.server.core.validators.SelfContainedTokenValidator;
import io.cellery.security.cell.sts.server.core.validators.TokenValidator;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.text.ParseException;
import java.util.HashMap;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import org.apache.commons.lang.StringUtils;
import org.apache.http.conn.ssl.DefaultHostnameVerifier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CelleryCellStsService {
    private static final Logger log = LoggerFactory.getLogger(CelleryCellStsService.class);
    private static final String CELLERY_AUTH_SUBJECT_CLAIMS_HEADER = "x-cellery-auth-subject-claims";
    private static final String KNATIVE_PROBE_HEADER_NAME = "k-network-probe";
    private static final TokenValidator TOKEN_VALIDATOR = new SelfContainedTokenValidator();
    protected static final String BEARER_HEADER_VALUE_PREFIX = "Bearer ";
    protected static final CellSTSRequestValidator REQUEST_VALIDATOR = new DefaultCellSTSReqValidator();
    protected static final AuthorizationService AUTHORIZATION_SERVICE = new AuthorizationService();
    protected UserContextStore userContextStore;
    protected UserContextStore localContextStore;

    public CelleryCellStsService(UserContextStore contextStore, UserContextStore localContextStore) throws CelleryCellSTSException {
        this.userContextStore = contextStore;
        this.localContextStore = localContextStore;
        this.setHttpClientProperties();
    }

    public void handleInboundRequest(CellStsRequest cellStsRequest, CellStsResponse cellStsResponse) throws CelleryCellSTSException {
        JWTClaimsSet jwtClaims;
        if (cellStsRequest.getRequestHeaders().containsKey(KNATIVE_PROBE_HEADER_NAME)) {
            log.debug("Ignoring knative probe request: {}:{} ", (Object)KNATIVE_PROBE_HEADER_NAME, (Object)cellStsRequest.getRequestHeaders().get(KNATIVE_PROBE_HEADER_NAME));
            return;
        }
        String requestId = cellStsRequest.getRequestId();
        try {
            boolean authenticationRequired = REQUEST_VALIDATOR.isAuthenticationRequired(cellStsRequest);
            if (!authenticationRequired) {
                return;
            }
            log.debug("Authentication is required for the request ID: {} ", (Object)requestId);
        }
        catch (CellSTSRequestValidationFailedException e) {
            throw new CelleryCellSTSException("Error while evaluating authentication requirement", e);
        }
        String callerCell = cellStsRequest.getSource().getCellInstanceName();
        log.debug("Caller cell : {}", (Object)callerCell);
        String jwt = this.getUserContextJwt(cellStsRequest);
        log.debug("Incoming JWT : " + jwt);
        if (CellStsUtils.isRequestToMicroGateway(cellStsRequest)) {
            log.debug("Request to micro-gateway intercepted");
            jwtClaims = this.handleRequestToMicroGW(cellStsRequest, requestId, jwt);
        } else if ("composite".equalsIgnoreCase(CellStsUtils.getMyCellName())) {
            if (StringUtils.isEmpty(jwt)) {
                log.debug("No token found in the request. Passing through from Composite STS");
                jwtClaims = null;
            } else {
                jwtClaims = this.handleRequestComposite(cellStsRequest, requestId, jwt);
            }
        } else {
            jwtClaims = this.handleInternalRequest(cellStsRequest, requestId, jwt);
        }
        try {
            AUTHORIZATION_SERVICE.authorize(cellStsRequest, jwt);
        }
        catch (AuthorizationFailedException e) {
            throw new CelleryCellSTSException("Authorization failure", e);
        }
        HashMap<String, String> headersToSet = new HashMap<String, String>();
        if (jwtClaims != null && StringUtils.isNotEmpty(jwtClaims.getSubject())) {
            headersToSet.put("x-cellery-auth-subject", jwtClaims.getSubject());
            log.debug("Set {} to: {}", (Object)"x-cellery-auth-subject", (Object)jwtClaims.getSubject());
        } else {
            headersToSet.put("x-cellery-auth-subject", "");
            log.debug("Subject is not available. No user context is passed.");
        }
        if (jwtClaims != null) {
            headersToSet.put(CELLERY_AUTH_SUBJECT_CLAIMS_HEADER, new PlainJWT(jwtClaims).serialize());
            log.debug("Set {} to : {}", (Object)CELLERY_AUTH_SUBJECT_CLAIMS_HEADER, (Object)new PlainJWT(jwtClaims).serialize());
        }
        cellStsResponse.addResponseHeaders(headersToSet);
    }

    private JWTClaimsSet handleRequestComposite(CellStsRequest cellStsRequest, String requestId, String jwt) throws CelleryCellSTSException {
        JWTClaimsSet jwtClaims;
        log.debug("Incoming request to composite STS from {}", (Object)cellStsRequest.getSource());
        try {
            log.debug("Validating incoming JWT {}", (Object)jwt);
            this.validateInboundToken(cellStsRequest, jwt);
            this.userContextStore.put(requestId, jwt);
            jwtClaims = this.extractUserClaimsFromJwt(jwt);
        }
        catch (TokenValidationFailureException e) {
            throw new CelleryCellSTSException("Error while validating JWT token", e);
        }
        return jwtClaims;
    }

    private JWTClaimsSet handleInternalRequest(CellStsRequest cellStsRequest, String requestId, String jwt) throws CelleryCellSTSException {
        JWTClaimsSet jwtClaims;
        log.debug("Call from a workload to workload within cell {} ; Source workload {} ; Destination workload {}", cellStsRequest.getSource().getCellInstanceName(), cellStsRequest.getSource().getWorkload(), cellStsRequest.getDestination().getWorkload());
        try {
            if (this.localContextStore.get(requestId) == null) {
                log.debug("Initial entrace to cell from gateway. No cached token found.");
                this.validateInboundToken(cellStsRequest, jwt);
                this.localContextStore.put(requestId, jwt);
            } else if (!StringUtils.equalsIgnoreCase(this.localContextStore.get(requestId), jwt)) {
                throw new CelleryCellSTSException("Intra cell STS token is tampered.");
            }
            jwtClaims = this.extractUserClaimsFromJwt(jwt);
        }
        catch (TokenValidationFailureException e) {
            throw new CelleryCellSTSException("Error while validating locally issued token.", e);
        }
        return jwtClaims;
    }

    protected JWTClaimsSet handleRequestToMicroGW(CellStsRequest cellStsRequest, String requestId, String jwt) throws CelleryCellSTSException {
        JWTClaimsSet jwtClaims;
        log.debug("Incoming request to cell gateway {} from {}", (Object)CellStsUtils.getMyCellName(), (Object)cellStsRequest.getSource());
        try {
            log.debug("Validating incoming JWT {}", (Object)jwt);
            this.validateInboundToken(cellStsRequest, jwt);
            this.userContextStore.put(requestId, jwt);
            jwtClaims = this.extractUserClaimsFromJwt(jwt);
        }
        catch (TokenValidationFailureException e) {
            throw new CelleryCellSTSException("Error while validating JWT token", e);
        }
        return jwtClaims;
    }

    private void validateInboundToken(CellStsRequest cellStsRequest, String token) throws TokenValidationFailureException {
        TOKEN_VALIDATOR.validateToken(token, cellStsRequest);
    }

    protected String getUserContextJwt(CellStsRequest cellStsRequest) {
        String authzHeaderValue = CellStsUtils.getAuthorizationHeaderValue(cellStsRequest.getRequestHeaders());
        return CellStsUtils.extractJwtFromAuthzHeader(authzHeaderValue);
    }

    public void handleOutboundRequest(CellStsRequest cellStsRequest, CellStsResponse cellStsResponse) throws CelleryCellSTSException {
        RequestDestination destination = cellStsRequest.getDestination();
        if (destination.isExternalToCellery()) {
            log.info("Intercepted an outbound call to a workload:{} outside Cellery. Passing the call through.", (Object)destination);
        } else {
            log.info("Intercepted an outbound call to a workload:{} within Cellery. Injecting a STS token for authentication and user-context sharing from Cell STS.", (Object)destination);
            this.attachToken(cellStsRequest, cellStsResponse);
        }
    }

    protected void attachToken(CellStsRequest cellStsRequest, CellStsResponse cellStsResponse) throws CelleryCellSTSException {
        String stsToken = this.getStsToken(cellStsRequest);
        if (StringUtils.isEmpty(stsToken)) {
            throw new CelleryCellSTSException("No JWT token received from the STS endpoint: " + CellStsConfiguration.getInstance().getStsEndpoint());
        }
        log.debug("Attaching jwt to outbound request : {}", (Object)stsToken);
        if (cellStsRequest.getRequestHeaders().get("x-cellery-auth-subject") != null) {
            log.info("Found user in outgoing request");
        }
        cellStsResponse.addResponseHeader("cellery-authorization", BEARER_HEADER_VALUE_PREFIX + stsToken);
    }

    private JWTClaimsSet extractUserClaimsFromJwt(String jwt) throws CelleryCellSTSException {
        if (StringUtils.isBlank(jwt)) {
            throw new CelleryCellSTSException("Cannot extract user context JWT from Authorization header.");
        }
        return this.getJWTClaims(jwt);
    }

    private JWTClaimsSet getJWTClaims(String jwt) throws CelleryCellSTSException {
        try {
            return SignedJWT.parse(jwt).getJWTClaimsSet();
        }
        catch (ParseException e) {
            throw new CelleryCellSTSException("Error while parsing the Signed JWT in authorization header.", e);
        }
    }

    private String getStsToken(CellStsRequest request) throws CelleryCellSTSException {
        String requestId = request.getRequestId();
        if (this.isRequestFromMicroGateway(request)) {
            log.debug("Request with ID: {} from micro gateway to {} workload of cell {}", requestId, request.getDestination().getWorkload(), request.getDestination().getCellName());
            if (StringUtils.isNotEmpty(this.localContextStore.get(requestId))) {
                log.debug("Found an already existing local token issued for same request on a different occurance");
                return this.localContextStore.get(requestId);
            }
            String jwt = this.userContextStore.get(requestId);
            if (StringUtils.isEmpty(jwt)) {
                return this.getTokenFromLocalSTS(CellStsUtils.getMyCellName(), request.getDestination().getWorkload());
            }
            return this.getTokenFromLocalSTS(jwt, CellStsUtils.getMyCellName(), request.getDestination().getWorkload());
        }
        if (!CellStsUtils.isCompositeSTS() && this.isIntraCellCall(request) && this.localContextStore.get(requestId) != null) {
            log.debug("Intra cell request with ID: {} from source workload {} to destination workload {} within cell {}", requestId, request.getSource().getWorkload(), request.getDestination().getWorkload(), request.getSource().getCellInstanceName());
            return this.localContextStore.get(requestId);
        }
        if (!CellStsUtils.isCompositeSTS() && !this.isIntraCellCall(request) && this.localContextStore.get(requestId) != null) {
            log.debug("Outbound call from home cell. Building token");
            String jwt = this.localContextStore.get(requestId);
            return this.getTokenFromLocalSTS(jwt, request.getDestination().getCellName(), request.getDestination().getWorkload());
        }
        if (CellStsUtils.isCompositeSTS() && this.userContextStore.get(requestId) != null) {
            String token = this.getTokenAsComposite(request, this.userContextStore.get(requestId));
            this.userContextStore.put(requestId, token);
            return token;
        }
        log.debug("Request initiated within cell {} to {}", (Object)request.getSource().getCellInstanceName(), (Object)request.getDestination().toString());
        String token = this.getUserContextJwt(request);
        if (StringUtils.isNotEmpty(token)) {
            log.debug("Found a token attached by the workload : {}", (Object)token);
            return this.getTokenWithWorkloadPassedBearerToken(request, token);
        }
        return this.getTokenFromLocalSTS(request.getDestination().getCellName(), request.getDestination().getWorkload());
    }

    private String getTokenWithWorkloadPassedBearerToken(CellStsRequest request, String token) throws CelleryCellSTSException {
        try {
            log.debug("Validating workload attached token.");
            TOKEN_VALIDATOR.validateToken(token, request);
            return this.getTokenFromLocalSTS(token, request.getDestination().getCellName(), request.getDestination().getWorkload());
        }
        catch (TokenValidationFailureException e) {
            throw new CelleryCellSTSException("Error while validating workload passed token", e);
        }
    }

    private String getTokenAsComposite(CellStsRequest request, String token) throws CelleryCellSTSException {
        log.debug("Issuing token as composite to outbound request");
        return this.getTokenFromLocalSTS(token, request.getDestination().getCellName(), request.getDestination().getWorkload());
    }

    private boolean isIntraCellCall(CellStsRequest cellStsRequest) throws CelleryCellSTSException {
        String currentCell = CellStsUtils.getMyCellName();
        String destinationCell = cellStsRequest.getDestination().getCellName();
        return StringUtils.equals(currentCell, destinationCell);
    }

    private boolean isRequestFromMicroGateway(CellStsRequest cellStsRequest) throws CelleryCellSTSException {
        String workload = cellStsRequest.getSource().getWorkload();
        return StringUtils.isNotEmpty(workload) && workload.startsWith(CellStsUtils.getMyCellName() + "--gateway");
    }

    protected String getTokenFromLocalSTS(String audience, String destination) throws CelleryCellSTSException {
        return STSTokenGenerator.generateToken(audience, CellStsUtils.getIssuerName(CellStsUtils.getMyCellName()), destination);
    }

    protected String getTokenFromLocalSTS(String jwt, String audience, String destination) throws CelleryCellSTSException {
        String token = STSTokenGenerator.generateToken(jwt, audience, CellStsUtils.getIssuerName(CellStsUtils.getMyCellName()), destination);
        log.info("Issued a token from local STS : " + CellStsUtils.getCellImageName());
        return token;
    }

    private void setHttpClientProperties() throws CelleryCellSTSException {
        CelleryTrustManager celleryTrustManager = new CelleryTrustManager();
        try {
            SSLContext sc = SSLContext.getInstance("SSL");
            sc.init(null, new TrustManager[]{celleryTrustManager}, new SecureRandom());
            HttpsURLConnection.setDefaultSSLSocketFactory(sc.getSocketFactory());
            HttpsURLConnection.setDefaultHostnameVerifier(new CelleryHostnameVerifier(new DefaultHostnameVerifier()));
        }
        catch (KeyManagementException | NoSuchAlgorithmException e) {
            throw new CelleryCellSTSException("Error while initializing SSL context");
        }
    }
}

